#!/usr/bin/env python
#######################################################################
# Copyright (C) 2013 - 2021  Seguesoft  Inc.                          #                                       
# Redistribution of this software, in whole or in part, is prohibited #        
# without the express written permission of Seguesoft.                #  
#######################################################################
   
# There three sets of GET commands:
# sget_*, hget_* and xget_*

# 1 sget_*: GET using Subtree filter
# Returns: resultTuple
# Where the resultTuple is a tuple of five elements:
# (SucceededFlag, ReplyXML, ReplyElem, RequestXML, RequestElem)                  
#
# When using sget_* the related YANG modules don't have to be loaded first. 

# 2 hget_*: High level GET
# Returns:  A tuple of (resultList, unknownList, resultTuple)
#           resultList is a list of tuple (IID, valueAndnsmapDict) 
#           unknownList is instance ID without corresponding loaded modules
#           resultTuple is the same as in sget_*
#
# All leafs excepting keys are included in the returned IID list.         
# All content matching node is removed from the reply (a normal reply contains
# the content match noded in the request
#
# If segueIID form is used the result will also use segueIID form
# A segueIID uses module name as prefix, rather than module prefix.
# Processing a segueIID does no need a prefix to namespace map argument
#
# When using hget_* the related YANG modules must be loaded first. 
#
# 3 xget_*:  GET using XPATH filter
# Returns: resultTuple
# Where the resultTuple is a tuple of five elements:
# (SucceededFlag, ReplyXML, ReplyElem, RequestXML, RequestElem)                  
#
#
# The usage of sget_config, hget_config, xget_config is similar, except that 
# you can specify an additional keyword argument
#
# source = 'candidate' | 'running' | 'startup'
# When source is 'unspecified' it defaults to 'running'

import os, sys


# Extend system path to import 'ssmanager' package under the installation 
# directory.
sys.path.append(os.path.join(os.path.dirname(os.path.abspath(__file__)) ,
                                         "..", "..", "ssclient"))
import ssmanager

# Create netconf session
s=ssmanager.create_session(host="seguesoft.com", port=830, 
                user="testconf", password="testconf2")

s.print_rpc_stdout(True)
#s.print_rpc_stdout(False)

# Send a <get> RPC using instance ID or schema node ID with namespace.")
# The result is a tuple of five values:
#   Operation succeeded (True or False), 
#   Reply in xml format, 
#   Reply in lxml element format, 
#   Request in xml format, 
#   Request in lxml element format

# If no prefix is used for a subID then assuming the same prefix as the closet subID 
r = s.sget("/ncm:netconf-state/schemas/schema/identifier",
           namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

r = s.sget("/ncm:netconf-state/schemas/schema[identifier='netopeer-cfgnetopeer']"
          "[version='2013-02-14'][format='yang']/location",
           namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

# Send a <get> RPC using instance ID or schema node ID  with multiple namespaces"
r = s.sget("/ncm:netconf-state/ncm:datastores/ncm:datastore", 
           "/if:interfaces/if:interface/if:name",
           namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring",
                       "if":"http://netconfcentral.org/ns/yuma-interfaces"}
           )

# Send a <get> RPC using instance ID or schema node ID without explicit namespace
r = s.sget("/netconf-state/datastores/datastore", "netconf-state/schemas/schema")

r = s.sget(("/netconf-state/datastores/datastore/name", "running"), 
             ("/netconf-state/schemas/schema/version","2010-10-04"))


# If you have a pre-built filter expression you can use it directly       
filterExpr="<if:interfaces xmlns:if='http://netconfcentral.org/ns/yuma-interfaces'> \
        <if:interface>                                                              \
          <if:name />                                                               \
          </if:interface>                                                           \
      </if:interfaces>"
r = s.sget(filter=filterExpr)
          
# Some special retrieval cases
print("\n#Send a <get> without filter. This is a special case")
r=s.sget()

# get-config example
# build subtree filter expression for you automatically
r = s.sget_config("/nacm:nacm",  namespaces={"nacm": "urn:ietf:params:xml:ns:yang:ietf-netconf-acm"})

if r[0] == True:    
    # extract a node value from reply using instance ID or schema node ID
    nacmEnabled = s.extract_value_from_rpc_reply(r[2], "/nacm:nacm/nacm:enable-nacm",
                            namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})
    print("######get nacm Enabled flag using extract_value_from_rpc_output_data ", nacmEnabled)
    
    # Convert RPC reply XML message or lxml element into a list of (instance ID or schema node ID, value) tuples
    resList = s.extract_prefixed_node_id_and_value_from_rpc_reply(r[2])    
    for res in resList:
        print("node ID ", res)

else:
    print("No config data received! ", r[1])      

r = s.sget_config("/nacm:nacm", source='candidate',
                namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

r = s.sget_config("/ncm:netconf-state/ncm:datastores/ncm:datastore", 
                    "/if:interfaces/if:interface/if:name",                
                     namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring",
                             "if": "http://netconfcentral.org/ns/yuma-interfaces"})

# Send a <get-config> RPC using instance ID or schema node ID without explicit namespace
# This example will not return any data because no config data in netconf-monitoring module
r = s.sget_config("/netconf-state/datastores/datastore", "netconf-state/schemas/schema")

# Send a <get-config> RPC using instance ID or schema node ID  and 
# a content matching subtree filter and with-default tag
r=s.sget_config(("/netconf-state/datastores/datastore/name", "running"), 
                ("/netconf-state/schemas/schema/version","2010-10-04"),     
                withDefaults="trim")

# Some special retrieval cases
r = s.sget_config()

# When using hget you must load related YANG modules first. The returned XML is
# turned into a list of Instance IDs based on the loaded modules

# set up YANG module path.  If no path is given, use the default path which is 
# the same as in NETCONFc GUI
  
# Set paths to YANG models and load them
# The YANG path directories MUST be writable by the application
# since the application will need to generate XML based 
# definition files.
#
#s.load_yang_modules_in_paths("C:/MY-YANG-modules",  "C:/MY-YANG-modules2", recursive=True)
# Set YANG model path to the default path, which is the same as in the GUI NETCONFc
s.load_yang_modules_in_paths()

# Using module prefix and a nsmap 
resLst, unknownLst, resTuple = s.hget("/ncm:netconf-state",
           namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

# When using module name as prefix, no nsmap is needed
resLst, unknownLst, resTuple = s.hget("/ietf-netconf-monitoring:netconf-state")
resLst, unknownLst, resTuple = s.hget("/ietf-netconf-monitoring:netconf-state/sessions/session/session-id")                                 

# Exclude content match nodes from the result    
resLst, unknownLst, resTuple = s.hget("/ncm:netconf-state/schemas/schema[identifier='ietf-netconf-monitoring']"
                        "[version = '2010-10-04'][format='yang']/identifier",
                        namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

# Keep content match nodes in the result    
resLst, unknownLst, resTuple = s.hget_return_all("/ncm:netconf-state/schemas/schema[identifier='ietf-netconf-monitoring']"
                        "[version = '2010-10-04'][format='yang']",
                        namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

for r in resLst:
    print("\nresult IID", r[0], " result IID value Dict", r[1])

#
# Send a <get> with xpath expression
#   
#filterExpr="/ncm:netconf-state/ncm:sessions/ncm:session[ncm:username='bob']"
filterExpr="/ncm:netconf-state"
r=s.xget(filterExpr, withDefaults='trim', 
                namespaces={"ncm": "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"})

print("Sending succeeded? ", 
                r[0], " \n=====Result=====:\n", 
                r[1], " \n=====Request=====:\n", r[3])

