# Here's a full drop-in replacement for your SSHSession that keeps all the method names, 
# properties, and parsing logic intact, but uses AsyncSSH internally. You can literally
# replace your old SSHSession with this class without changing the rest of your code.
import os, socks, socket
import asyncio
import asyncssh
from io import BytesIO
from select import select
import logging
import traceback
from queue import Queue, Empty


#from aiohttp_socks import open_connection

from .errors import AuthenticationError, SessionCloseError, SSHError, SSHUnknownHostError
from .session import Session
from .session import HelloHandler    # if HelloHandler.build is in session module; adjust import to where HelloHandler.build lives


BUF_SIZE = 4096
MSG_DELIM = "]]>]]>"
TICK = 0.1
MSG_DELIM_2 = "\n##\n"

KH_FILE_NAME="~/.ssh/known_hosts"

DIGIT=['0','1','2','3','4','5','6','7','8','9']
DIGIT1=['1','2','3','4','5','6','7','8','9']


def default_unknown_host_cb(host, fingerprint, key):
    """An unknown host callback returns `True` if it finds the key acceptable, and `False` if not.

    This default callback always returns `False`, which would lead to :meth:`connect` raising a :exc:`SSHUnknownHost` exception.
    
    Supply another valid callback if you need to verify the host key programatically.

    *host* is the hostname that needs to be verified

    *fingerprint* is a hex string representing the host key fingerprint, colon-delimited e.g. `"4b:69:6c:72:6f:79:20:77:61:73:20:68:65:72:65:21"`
    """
    return False

def _colonify(fp):
    finga = fp[:2].decode("utf-8") 
    for idx  in range(2, len(fp), 2):
        finga += ":" + fp[idx:idx+2].decode("utf-8") 
    return finga

logger = logging.getLogger(__name__)
    
def run_in_main_thread(coro):
    """
    Ensure the coroutine runs in the main thread.
    Thread-safe.
    """
    try:
        loop = asyncio.get_event_loop()
    except RuntimeError:
        # no event loop in this thread -> create new
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)

    if loop.is_running():
        # Already running main loop (typical for asyncio apps)
        future = asyncio.run_coroutine_threadsafe(coro, loop)
        return future.result()
    else:
        return loop.run_until_complete(coro)
    
# ------------------- Channel Wrapper -------------------
class AsyncSSHChannelWrapper:
    """Wrap AsyncSSH channel for synchronous recv/send calls."""
    def __init__(self, chan: asyncssh.SSHClientChannel):
        self._chan = chan
        self._loop = asyncio.get_event_loop()
        self._buffer = bytearray()
        self._closing = False

        # link to subsystem (so data_received can append)
        if hasattr(chan, 'session'):
            chan.session._wrapper = self

        # start background reader task
        asyncio.create_task(self._reader())

    async def _reader(self):
        try:
            while not self._chan.is_closing():
                data = await self._chan.read(BUF_SIZE)
                if data:
                    if isinstance(data, str):
                        data = data.encode('utf-8')
                    self._buffer.extend(data)
                else:
                    self._closing = True
                    break
        except Exception:
            self._closing = True

    def recv(self, n):
        # synchronous-like read for old code
        while len(self._buffer) == 0 and not self._closing:
            asyncio.run(asyncio.sleep(0.01))
        data = self._buffer[:n]
        self._buffer = self._buffer[n:]
        return bytes(data)

    def write(self, data):
        # AsyncSSH expects str
        if isinstance(data, bytes):
            data = data.decode('utf-8')
        fut = self._chan.write(data)
        # optional: ensure it is scheduled
        return fut

    def is_closing(self):
        return self._closing
    

"""
class AsyncSSHChannelWrapper:
    def __init__(self, channel):
        self._channel = channel
        try:
            self._loop = asyncio.get_event_loop()
        except RuntimeError:
            self._loop = asyncio.new_event_loop()
            asyncio.set_event_loop(self._loop)

    def recv(self, n):
        # read is async, so we wrap it
        fut = asyncio.run_coroutine_threadsafe(self._channel.read(n), self._loop)
        return fut.result()

    def send(self, data):
        # write is synchronous, just call directly
        if isinstance(data, bytes):
            data = data.decode('utf-8')
        return self._channel.write(data)

    def write(self, data):
        return self.send(data)

    def is_closing(self):
        return self._channel.is_closing()

"""
class SSHSession(Session):
    def __init__(self, capabilities, timeout=60):
        super().__init__(capabilities, timeout=timeout)
        self._host = "localhost"
        self._hostsrc = "localhost"
        self._conn = None           # AsyncSSH connection
        self._channel = None        # NETCONF subsystem channel
        self._connected = False
        self._buffer = BytesIO()
        self._parsing_state = 0
        self._parsing_pos = 0
        self._delim_ver = 1
        self.chunk_msg_test_buffer_len = 0
        self._q = Queue()
    
    """
    async def _async_run(self):
        #Main async read/write loop.
        #Continuously reads from the NETCONF SSH channel and sends queued messages.
        try:
            while True:
                if not self._channel or self._channel.is_closing():
                    await asyncio.sleep(TICK)
                    continue

                # ------------------------------------------------------------
                # 1. READ FROM DEVICE (CORRECT AsyncSSH call)
                # ------------------------------------------------------------
                try:
                    data = await self._channel.recv(BUF_SIZE)
                except Exception as e:
                    logger.error(f"AsyncSSH: recv error: {e}")
                    break

                if data:
                    if isinstance(data, str):
                        data = data.encode()

                    self._buffer.write(data)

                    try:
                        if self._delim_ver == 1:
                            self._parse()
                        else:
                            self._parse2()
                    except Exception as e:
                        logger.error(f"AsyncSSH parse error: {e}")

                # ------------------------------------------------------------
                # 2. SEND QUEUED MESSAGES
                # ------------------------------------------------------------
                try:
                    msg = self._q.get_nowait()

                    if self._delim_ver == 1:
                        # NETCONF 1.0 delimiter
                        frame = msg.decode('utf-8') + MSG_DELIM
                        await self._channel.send(frame)
                    else:
                        # NETCONF 1.1 (RFC6242 chunk)
                        b = msg if isinstance(msg, bytes) else msg.encode()
                        chunk = f"{len(b)}\n".encode() + b + b"\n##\n"
                        await self._channel.send(chunk)

                except Empty:
                    await asyncio.sleep(TICK)
                except Exception as e:
                    logger.error(f"AsyncSSH send error: {e}")
                    break

        except asyncio.CancelledError:
            logger.info("AsyncSSH run loop cancelled")
        except Exception as e:
            logger.error(f"AsyncSSH run loop exception: {e}", exc_info=True)

        finally:
            await self.close()
            logger.info("AsyncSSH run loop terminated")

    """
    async def _async_run(self):
        
        #Async loop for reading from and writing to the AsyncSSH channel.
        #Handles queued messages in self._q and parses incoming data.
        try:
            while self._connected:
                # ----- READ incoming data -----
                if self._channel is None or self._channel.is_closing():
                    await asyncio.sleep(TICK)
                    continue

                try:
                    data = await self._channel.read(BUF_SIZE)
                except Exception as e:
                    # channel may be closed
                    break

                if data:
                    # data might be str or bytes; ensure bytes for buffer
                    if isinstance(data, str):
                        data = data.encode('utf-8')
                    self._buffer.write(data)

                    # parse buffer
                    if self._delim_ver == 1:
                        self._parse()
                    else:
                        self._parse2()

                # ----- WRITE queued messages -----
                try:
                    while True:
                        msg = self._q.get_nowait()
                        # Ensure msg is bytes
                        if isinstance(msg, str):
                            msg_bytes = msg.encode('utf-8')
                        else:
                            msg_bytes = msg

                        # Frame message depending on delim version
                        if self._delim_ver == 1:
                            await self._channel.write(msg_bytes + MSG_DELIM.encode('utf-8'))
                        else:
                            # If chunked framing, handle appropriately
                            await self._channel.write(msg_bytes)

                except Empty:
                    # no messages to send
                    await asyncio.sleep(TICK)

        except Exception as e:
            logger.debug("AsyncSSH run loop error: %r", e)
            traceback.print_exc()
        finally:
            await self.close()


    async def _async_connect(self, host, port=830, username=None, password=None,
                            key_filename=None, socks_proxy=None,
                            known_hosts=None, ciphers=None):
        """AsyncSSH connection with optional SOCKS proxy."""
        sock = None

        # ----- SOCKS PROXY -----
        if socks_proxy and socks_proxy.get("server"):
            sock = socks.socksocket(socket.AF_INET, socket.SOCK_STREAM)
            proxy_type = socks.SOCKS5 if socks_proxy.get("type", "SOCKS5") == "SOCKS5" else socks.SOCKS4
            sock.set_proxy(proxy_type,
                        addr=socks_proxy["server"],
                        port=socks_proxy.get("port", 1080),
                        username=socks_proxy.get("user"),
                        password=socks_proxy.get("password"))
            sock.connect((host, port))
            sock.setblocking(False)

        # ----- CIPHERS -----
        if ciphers is None:
            ciphers = [
                "chacha20-poly1305@openssh.com",
                "aes256-gcm@openssh.com",
                "aes128-gcm@openssh.com",
                "aes256-ctr",
                "aes192-ctr",
                "aes128-ctr"
            ]

        client_keys = [key_filename] if key_filename else None

        # ----- SSH CONNECT -----
        try:
            self._conn = await asyncssh.connect(
                host if not sock else None,
                port=port,
                username=username,
                password=password,
                client_keys=client_keys,
                known_hosts=known_hosts,
                encryption_algs=ciphers,
                sock=sock
            )
        except Exception as e:
            raise SSHError(f"Failed to connect: {e}")

        # ----- OPEN NETCONF SUBSYSTEM (THE CORRECT WAY) -----
        # open NETCONF subsystem correctly        
        # wrap the AsyncSSH channel so ncclient can call recv/send/is_closing
        chan, session = await self._conn.create_session(lambda: AsyncSSHSubsystem(), subsystem='netconf')
        self._channel = AsyncSSHChannelWrapper(chan)
        # now send client hello
        hello_msg = HelloHandler.build(self._client_capabilities)  # your old hello builder

        # Correctly handle both str and bytes
        if isinstance(hello_msg, str):
            data_to_send = hello_msg.encode('utf-8') + MSG_DELIM.encode('utf-8')
        else:
            data_to_send = hello_msg + MSG_DELIM.encode('utf-8')

        if isinstance(data_to_send, bytes):
            data_to_send = data_to_send.decode('utf-8')
        self._channel.write(data_to_send)

        # ----- START RUN LOOP -----
        if not hasattr(self, "_reader_task"):
            self._reader_task = asyncio.create_task(self._async_run())

        # ----- SEND CLIENT HELLO -----
        hello = HelloHandler.build(self._client_capabilities)
        self._q.put(hello.encode('utf-8'))

        self._host = host
        try:
            self._hostsrc = self._conn.get_extra_info('sockname')[0]
        except Exception:
            self._hostsrc = None
        self._connected = True


    # Synchronous wrapper to match your old connect signature
    def connect(self, host, port=830, timeout=None, unknown_host_cb=None,
                username=None, password=None, key_filename=None, allow_agent=False,
                look_for_keys=False, client_sock=None,
                socks_proxy={"server":'', "port":1080, "type":"SOCKS5", "user": '', "password":''},
                preferred_ciphers=None, known_hosts=None):

        if timeout is not None:
            self.timeout = timeout

        # Determine known_hosts file if not provided
        if known_hosts is None:
            known_hosts = os.path.expanduser(KH_FILE_NAME)
            if not os.path.exists(known_hosts):
                known_hosts = None  # skip verification

        run_in_main_thread(
            self._async_connect(host, port, username, password,
                                key_filename, socks_proxy, known_hosts,
                                preferred_ciphers)
        )



    def close(self):
        if self._conn:
            try:
                run_in_main_thread(self._conn.wait_closed())
            except Exception:
                pass
            self._conn.close()
        self._connected = False


    @property
    def transport(self):
        # Provide an object with set_keepalive(n) for compatibility
        class TransportWrapper:
            def __init__(self, conn):
                self._conn = conn

            def set_keepalive(self, n):
                # AsyncSSH does not use keepalive in the same way
                pass

        return TransportWrapper(self._conn)
    
    # ------------------ Keepalive ------------------
    @property
    def keep_alive(self):
        return self._keep_alive

    @keep_alive.setter
    def keep_alive(self, interval):
        self._keep_alive = interval
        if self._keep_alive_task:
            self._keep_alive_task.cancel()
        if self._connected and interval:
            self._keep_alive_task = asyncio.create_task(self._keepalive_loop(interval))

    async def _keepalive_loop(self, interval):
        while self._connected:
            try:
                if self._channel:
                    await self._channel.send_ignore()
            except Exception:
                pass
            await asyncio.sleep(interval)

    # ------------------ Transport property for backward compatibility ------------------
    @property
    def _transport(self):
        """
        Keep _transport property for old code compatibility.
        """
        return self

    def set_keepalive(self, interval):
        """
        Dummy set_keepalive() to support old code.
        """
        self.keep_alive = interval


    # --- Synchronous run loop (using AsyncSSH channel) ---
    def run(self):
        chan = self._channel
        q = self._q

        try:
            while True:
                r, w, e = select([chan], [], [], TICK)
                if r:
                    data = chan.read_nowait(BUF_SIZE)
                    if data:
                        self._buffer.write(data.encode() if isinstance(data, str) else data)
                        if self._delim_ver == 1:
                            self._parse()
                        else:
                            self._parse2()
                    else:
                        raise SessionCloseError(self._buffer.getvalue().decode())
                if not q.empty() and chan.write_ready():
                    data = q.get().decode("utf-8") + MSG_DELIM
                    while data:
                        n = chan.write(data)
                        if n <= 0:
                            raise SessionCloseError(self._buffer.getvalue(), data)
                        data = data[n:]
        except Exception as e:
            logger.debug("Broke out of main loop, error=%r", e)
            self._dispatch_error(repr(e) + traceback.format_exc())
            self.close()

    async def send_message(self, msg):
        if self._channel:
            await self._channel.write(msg)

    async def send_message(self, msg):
        if self._channel:
            await self._channel.write(msg)

    # -----------------------------------------------------------------
    # COPY your existing _parse(), _parse2(), _readchunk() methods here
    # -----------------------------------------------------------------
    #def _parse(self):
    #    # paste your Paramiko-based _parse() here
    #    pass

    #def _parse2(self):
    #    # paste your Paramiko-based _parse2() here
    #    pass

    #def _readchunk(self, x, buf, rpcmsg):
    #    # paste your Paramiko-based _readchunk() here
    #    pass

    def _parse(self):
        "Messages ae delimited by MSG_DELIM. The buffer could have grown by a maximum of BUF_SIZE bytes everytime this method is called. Retains state across method calls and if a byte has been read it will not be considered again."
        delim = MSG_DELIM
        n = len(delim) 
        expect = self._parsing_state
        buf = self._buffer
        buf.seek(self._parsing_pos)
        while True:
            x = buf.read(1)
            #print("x got ", x)
            if not x: # done reading
                break
            elif x.decode() == delim[expect]: # what we expected
                expect += 1 # expect the next delim char
            else:
                expect = 0
                continue
            # loop till last delim char expected, break if other char encountered
            for i in range(expect, n):
                x = buf.read(1)
                if not x: # done reading
                    break
                if x.decode() == delim[expect]: # what we expected
                    #print("expect ", expect, "got ", x) 
                    expect += 1 # expect the next delim char                    
                else:
                    expect = 0 # reset
                    break
            else: # if we didn't break out of the loop, full delim was parsed
                #print('Parsed a new message:\n')
                msg_till = buf.tell() - n
                buf.seek(0)                
                newmsg = buf.read(msg_till).strip()
                logger.info(newmsg.decode())

                self._dispatch_message(newmsg.decode())
                    
                # Python3 In text files (those opened without a b 
                # in the mode string),
                # only seeks relative to the beginning of the file 
                # [os.SEEK_SET] are allowed...
                #buf.seek(n, os.SEEK_CUR)
                buf.seek(buf.tell() + n, os.SEEK_SET) 

                rest = buf.read()
                #close current buf
                buf.close()
                #create a new buf to hold data
                buf = BytesIO()
                buf.write(rest)
                buf.seek(0)
                
                expect = 0
                
        self._buffer = buf
        self._parsing_state = expect
        self._parsing_pos = self._buffer.tell()
    
    def _parse2(self):
        rpcmsg=b""
        buf = self._buffer        
        # test data 
        '''   
        testData= buf.getvalue()
        if len(testData.decode()) > 100 and testData.decode().find("ietf-yang-library") == -1:
            data="""
#982
<?xml version="1.0" encoding="UTF-8"?><rpc-reply message-id="b2e023be-5d82-11ec-bd2b-00133b190f3c" xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"><data xmlns="urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring">
module ieee802-dot1q-bridge-sched {
  namespace urn:ieee:std:802.1Q:yang:ieee802-dot1q-bridge-sched;
  prefix bridge-sched;

  import ietf-interfaces {
    prefix if;
  }
  import ieee802-dot1q-bridge {
    prefix dot1q;
  }

  import ieee802-dot1q-sched {
    prefix sched;
  }
  organization
    "IEEE 802.1 Working Group";
  contact
    "WG-URL: http://www.ieee802.org/1/
    WG-EMail: stds-802-1-l@ieee.org

    Contact: IEEE 802.1 Working Group Chair
    Postal: C/O IEEE 802.1 Working Group
    IEEE Standards Association
    445 Hoes Lane
    Piscataway, NJ 08854
    USA

    E-mail: STDS-802-1-CHAIRS@IEEE.ORG";
  description
    "This module provides for management of IEEE Std 802.1Q Bridges
    that support Scheduled Traffic Enhancements.";

  revisio
#519
n 2021-02-02 {
    description
     "Published as part of IEEE Std 802.1Qcw.
      Initial version.";
    reference
      "IEEE Std 802.1Qcw - Bridges and Bridged Networks — Amendment:
      YANG Data Models for Scheduled Traffic, Frame Preemption, and
      Per-Stream Filtering and Policing.";
  }


  augment "/if:interfaces/if:interface/dot1q:bridge-port" {

    description
      "Augment bridge-port with Scheduled Traffic configuration.";

   uses sched:sched-parameters;
  }
}
</data></rpc-reply>
##
"""       
            buf =BytesIO(data.encode())      
        '''

        # Done with test data          
        buf.seek(0)
        while True:
            x = buf.read(1)   
            #print("x ", x)
            if  len(x)==0: #EOF
                #print "break "
                break
            elif x.decode() == '\n': 
                #print("x1 " , x)
                x = buf.read(1)
                if len(x)==0: # EOF
                    break                
                elif x.decode() == '#': # maybe chunk start "\n#12345\n"
                    #print("x2 " , x)
                    #read in chunksize 1st number
                    x = buf.read(1)                     
                    if len(x)==0: #EOF
                        #print("break2 ")
                        break
                    elif x.decode() in DIGIT1:
                        rpcmsg=self._readchunk(x, buf,rpcmsg)
                    elif x.decode() == '#':
                        #possible end-of-chunk-mark
                        #print("MAY be end of mark")
                        x = buf.read(1)                            
                        if len(x)==0: #EOF
                            break
                        elif x.decode() == '\n': #end of chunks                     
                            #confirmed endofchunks mark encountered
                            # debugging reply
                            #print("parse2 got: ", rpcmsg.decode())
                            #if rpcmsg.find("hello") == -1:
                            #    rpcmsg="""<?xml version='1.0' encoding='UTF-8'?>
                            #    <rpc-reply xmlns:ncx="http://netconfcentral.org/ns/yuma-ncx" xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
                            #    <rpc-error>
                            #      <error-type>transport</error-type>
                            #      <error-tag>malformed-message</error-tag>
                            #      <error-severity>error</error-severity>
                            #      <error-app-tag>data-invalid</error-app-tag>
                            #      <error-message xml:lang="en">invalid protocol framing characters received</error-message>
                            #    </rpc-error>
                            #    </rpc-reply>"""
                            
                            self._dispatch_message(rpcmsg.decode())
                            #preserve the remaining data
                            rest = buf.read()
                            #print("rest data: ", rest)
                            #close current buf
                            buf.close()
                            #create a new buf to hold the remaining data
                            buf = BytesIO()
                            buf.write(rest)
                            buf.seek(0)
                            #process the next msg
                            continue
                    else:
                        #print("not matched x ", x)
                        #buf.seek(-2, os.SEEK_CUR)
                        buf.seek(buf.tell() -2, os.SEEK_SET) # os.SEEK_SET == 0
                else:
                    #print("not matched x1 ", x)
                    #buf.seek(-1, os.SEEK_CUR)
                    buf.seek(buf.tell() -1, os.SEEK_SET) # os.SEEK_SET == 0   
                    
        #assign session buffer to the newly created one if that happened                
        self._buffer = buf 

    def _readchunk(self, x, buf, rpcmsg):
        # it is char 1-9
        size=x
        #print("found size digit  ", x)
        # parse out the remaining size digits

        while True:
            x=buf.read(1)
            if not x:
                break
            elif x.decode() in DIGIT:
                #print("found size digit  ", x)
                size += x #append another size digit
            elif x.decode() == "\n":   
                #chunk size ends
                intsize =int(size)
                #print("found new real size   ", intsize)
                #make sure the intsize does not overflow
                if intsize > 4294967295:
                    #invalid size, terminate the session
                    logger.debug('chunk size invalid > 4294967295')
                    self.close()
                    raise SessionCloseError("invalid chunksize %s"%str(intsize))                                    
                    
                else:
                    #read this chunk in
                    #print("intsize", intsize)
                    msg_till = buf.read(intsize)
                    #print("msg_till ", msg_till, "len(msg_till) ", len(msg_till))
                    if not msg_till or len(msg_till) != intsize:
                        #print("NOT Enough DATA!!!")
                        # logger.debug("NOT Enough DATA read in %d", len(msg_till))
                        #not enough data to read we need to accept in more data in socket layer
                        break
                    else:
                        rpcmsg += msg_till
                        
                        #check if this chunk contents have been read out completely       
                        x=buf.read(1)
                        #print("x3 : ", x.decode())
                        if not x:
                            break #EOF data not complete for a whole message
                        elif x.decode() =='\n':
                            x=buf.read(1)
                            if not x:
                                break #data not complete for a whole message
                            elif x.decode() =='#':
                                #print("got  all data for a chunk")
                                #reset read pointer for another chunk 
                                # and end-of-check-mark
                                #print "YES next chunk or end"
                                #buf.seek(-2, os.SEEK_CUR)
                                buf.seek(buf.tell()-2, os.SEEK_SET)
                                break                                            
                        else:
                            #print("unexpected chunk framing: ", x)
                            logger.debug('The length of the chunk does not match the chunk size value! Data remains...')
                            self.close()
                            #raise SessionCloseError("The length of the chunk does not match the chunk size value! rpcmsg read: " +
                            #    rpcmsg.decode() + " \nData remains... All msg: " + buf.getvalue().decode() )                            
                            raise SessionCloseError(b"The length of the chunk does not match the chunk size value! rpcmsg read: " +
                                rpcmsg + b" \nData remains... \nAll msg: " + buf.getvalue())                                    
        return rpcmsg
            
    def load_known_hosts(self, filename=None):
        """Load host keys from an openssh :file:`known_hosts`-style file. Can be called multiple times.

        If *filename* is not specified, looks in the default locations i.e. :file:`~/.ssh/known_hosts` and :file:`~/ssh/known_hosts` for Windows.
        
        """
        
        if filename is None:
            if os.pathsep == ";":
                filename = os.path.expanduser(KH_FILE_NAME)              
            else: 
                filename = os.path.expanduser(KH_FILE_NAME)                 

            #print("userssh filename ", filename, " isfile ", os.path.isfile(filename))
            try:
                self._host_keys.load(filename)
            except Exception:
                pass             
        else:
            self._host_keys_filename = filename             
            self._host_keys.load(filename)

    def save_host_keys(self, filename=None):
        if filename is None:
           if os.pathsep == ";":
               filename = os.path.expanduser(KH_FILE_NAME)              
           else: 
               filename = os.path.expanduser(KH_FILE_NAME)              
        #print("dir ", os.path.dirname(filename))
        os.makedirs(os.path.dirname(filename), exist_ok=True)

        f = open(filename, 'w') 
        f.write('# SSH host keys collected by paramiko\n') 
        for hostname, keys in self._host_keys.items(): 
          for keytype, key in keys.items(): 
              f.write('%s %s %s\n' % (hostname, keytype, key.get_base64())) 
        f.close() 
        


    # REMEMBER to update transport.rst if sig. changes, since it is hardcoded there
    # paramiko no existing session exception
    # As you already have password you don't need to talk to agent or look for private keys stored on your machine. 
    # So try passing extra parameters allow_agent, look_for_keys:
    

class AsyncSSHSubsystem(asyncssh.SSHClientSession):
    def __init__(self):
        self._wrapper = None

    def data_received(self, data, datatype):
        if self._wrapper:
            if isinstance(data, str):
                data = data.encode('utf-8')
            self._wrapper._buffer.extend(data)

    #def data_received(self, data, datatype):
    #    if self._wrapper:
    #        self._wrapper.append_data(data.encode() if isinstance(data, str) else data)

    def connection_lost(self, exc):
        if self._wrapper:
            self._wrapper._closing = True
