#---------------------------------------------------------------------
# Copyright (C) 2013  Seguesoft  Inc.
#                                                                             
# Redistribution of this software, in whole or in part, is prohibited         
# without the express written permission of Seguesoft. 
# Modified based on ncclient
# ----------------------------------------------------------------------
# Copyright 2009 Shikhar Bhushan
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"Methods for creating, parsing, and dealing with XML and ElementTree objects."

#from io import StringIO
from io import BytesIO
#iterparse works with file like objects that return bytes. 
#from xml.etree import cElementTree as ET
from lxml import etree as ET

from . import NCClientError
class XMLError(NCClientError): pass


### Namespace-related

#: Base NETCONF namespace
BASE_NS_1_0 = "urn:ietf:params:xml:ns:netconf:base:1.0"
NO_BASE_NS_1_0 = False # for working around some server that does not place rpc in the BASE_NS_1_0 namespace
#: Namespace for Tail-f core data model
TAILF_AAA_1_1 = "http://tail-f.com/ns/aaa/1.1"
#: Namespace for Tail-f execd data model
TAILF_EXECD_1_1 = "http://tail-f.com/ns/execd/1.1"
#: Namespace for Cisco data model
CISCO_CPI_1_0 = "http://www.cisco.com/cpi_10/schema"
#: Namespace for Flowmon data model
FLOWMON_1_0 = "http://www.liberouter.org/ns/netopeer/flowmon/1.0"

NC_MONITORING= "urn:ietf:params:xml:ns:yang:ietf-netconf-monitoring"
#with-defaults 
WITH_DEFAULT_1_0 = "urn:ietf:params:xml:ns:yang:ietf-netconf-with-defaults"
try:
    register_namespace = ET.register_namespace
except AttributeError:
    def register_namespace(prefix, uri):
        from xml.etree import ElementTree
        # cElementTree uses ElementTree's _namespace_map, so that's ok
        ElementTree._namespace_map[uri] = prefix
#register_namespace.func_doc = "ElementTree's namespace map determines the prefixes for namespace URI's when serializing to XML. This method allows modifying this map to specify a prefix for a namespace URI."

for (ns, pre) in list({
    BASE_NS_1_0: 'nc',
    TAILF_AAA_1_1: 'aaa',
    TAILF_EXECD_1_1: 'execd',
    CISCO_CPI_1_0: 'cpi',
    FLOWMON_1_0: 'fm',
    WITH_DEFAULT_1_0: 'ncwd',
}.items()): register_namespace(pre, ns)

#qualify = lambda tag, ns=BASE_NS_1_0: tag if ns is None else "{%s}%s" % (ns, tag)
"""Qualify a *tag* name with a *namespace*, in :mod:`~xml.etree.ElementTree` fashion i.e. *{namespace}tagname*."""

def qualify(tag, ns=BASE_NS_1_0):
    if ns is None or NO_BASE_NS_1_0:
        return tag
    else:    
        return "{%s}%s" % (ns, tag)

def to_xml(ele, encoding="utf-8", xmldeclaration=True, pretty_print=True):
    "Convert and return the XML for an *ele* (:class:`~xml.etree.ElementTree.Element`) with specified *encoding*."
          
    xml = ET.tostring(ele, encoding=encoding,  xml_declaration=xmldeclaration, pretty_print=pretty_print)
    #if xmldeclaration ==True:
    #    xml= xml if xml.startswith('<?xml') else '<?xml version="1.0" encoding="%s"?>%s' % (encoding, xml)
    #else:
    #    xmlidx=xml.find("?>")
    #    xml=xml[xmlidx+2:]
    return xml

def to_ele(x):
    "Convert and return the :class:`~xml.etree.ElementTree.Element` for the XML document *x*. If *x* is already an :class:`~xml.etree.ElementTree.Element` simply returns that."        
    if ET.iselement(x):
        return x
        
    else:
        try:
            #return ET.fromstring(x)
            return ET.fromstring(x.encode("utf-8"))
        except (UnicodeDecodeError, UnicodeEncodeError, AttributeError):
            return ET.fromstring(x.decode('utf-8'))

        
def parse_root(raw):
    "Efficiently parses the root element of a *raw* XML document, returning a tuple of its qualified name and attribute dictionary."
    # In Python 2.7  StringIO module was capable handling the Byte as well Unicode
    # But in python3 you will have to use separate BytesIO for handling Byte strings
    # and StringIO for handling Unicode strings.

    # iterparse works with file like objects that return bytes. 
    # BytesIO("<root><a>data</a></root>".encode('utf-8'))
    #fp = StringIO(raw)
    fp = BytesIO(raw.encode('utf-8')) 
    for event, element in ET.iterparse(fp, events=('start',)):
        return (element.tag, element.attrib)

def validated_element(x, tags=None, attrs=None):
    """Checks if the root element of an XML document or Element meets the supplied criteria.
    
    *tags* if specified is either a single allowable tag name or sequence of allowable alternatives

    *attrs* if specified is a sequence of required attributes, each of which may be a sequence of 
            several allowable alternatives

    Raises :exc:`XMLError` if the requirements are not met.
    """
    ele = to_ele(x)
    if tags:
        if isinstance(tags, str):
            tags = [tags]
        if ele.tag not in tags:
            raise XMLError("Element [%s] is not the expected element: %s" % (ele.tag, tags))
    if attrs:
        for req in attrs:
            if isinstance(req, str): req = [req]
            for alt in req:
                if alt in ele.attrib:
                    break
            else:
                raise XMLError("Element [%s] does not have required attributes: %s" % (ele.tag, req))
    return ele

#We need to change to use a non lamba, so we can clear extra and attrs
#otherwise unexpected attrs are added to rpc . reason unknow yet. XL    
#new_ele = lambda tag, attrs={}, **extra:  ET.Element(tag, attrs, **extra)
def new_ele(tag, attrs={}, **extra): 
    #print "tag: ", tag, " attrsss", attrs, " extra", extra
    elm=ET.Element(tag, attrs, **extra)
    extra.clear()
    attrs.clear()
    return elm
    
#sub_ele = lambda parent, tag, attrs={}, **extra: ET.SubElement(parent, tag, attrs, **extra)
def sub_ele(parent, tag, attrs={}, **extra):
    #print "sub tag: ", tag, " attrsss", attrs, " extra", extra
    elm=ET.SubElement(parent, tag, attrs, **extra)
    extra.clear()
    attrs.clear()
    return elm
