######################################################################
# Copyright (C) 2011 - 2014 Seguesoft Inc. All rights reserved.
# Redistribution of this software, in whole or in part, is prohibited         
# without the express written permission of Seguesoft. 
######################################################################


from common_global_variables import DEBUGPRINT
import sys
from lxml import etree as ElementTree
from parse_module import SS_YANG_Dict
#sys.path.append(os.path.join(os.path.dirname(os.path.abspath(__file__)) , ".."))
#from ncclient.operations import util

  
def _construct_namespace_qualified_name(elementName, prefix_namespace_dict):
    #In the elementName passed in does not have a prefix qualifier, then
    #the same elementName will be returned.
    idx = elementName.find(":")
    if idx != -1:
        elementNamePfx=elementName[0:idx]
        subelnameName=elementName[idx+1:]
        if elementNamePfx in prefix_namespace_dict:
            subelementNamespace= prefix_namespace_dict[elementNamePfx]
            #create node with augmenting namespace
            elementName = "{%s}%s"%(subelementNamespace,subelnameName)
        else:
            # assuming module identifier is used            
            if  elementNamePfx in SS_YANG_Dict:
                subelementNamespace = SS_YANG_Dict[elementNamePfx]["namespace"]
                elementName = "{%s}%s"%(subelementNamespace,subelnameName)
            else:    
                raise NamespaceMissingError(elementNamePfx)                       
    return elementName     

def _constrcut_element_namespace_qualifier(pfx, ns):
    ElementTree.register_namespace(pfx, ns)  
    nsQualifier="{%s}" % ns
    return nsQualifier

nsQualifierBase = _constrcut_element_namespace_qualifier("xc", "urn:ietf:params:xml:ns:netconf:base:1.0")                        
nsQualifieWithDefaults = _constrcut_element_namespace_qualifier("wd", "urn:ietf:params:xml:ns:netconf:default:1.0")                        
nsQualifierYang = _constrcut_element_namespace_qualifier("yi", "urn:ietf:params:xml:ns:yang:1")
    
def _get_data_rpc_helper(m, expr, withDefaults="", xpath=False, xPathNamespacePrefixMap=None, **kwargs):
    """
    m: ncclient manager instance
    withDefaults: explicit, report-all, trim, report-all-tagged
    """
    result=""    
    if xpath == False:
        if expr == None:    
            #no filter
            result=m.get_data(withDefaults=withDefaults, **kwargs)                          
        else:   
            # expr passed in may be bytes or str so we use duct tape here
            try:
                expr = expr.decode("utf-8")
            except AttributeError:
                pass

            if expr.strip().startswith("<subtree-filter"):
                #already a filter str that includes <filter> element
                result=m.get_data(filter=expr, withDefaults=withDefaults, **kwargs)
            else:
                result=m.get_data(filter=('subtree-filter', expr), withDefaults=withDefaults, **kwargs)
    else:
        if expr == None:      
                result=m.get_data(withDefaults=withDefaults, **kwargs)                
        else:   
                result=m.get_data(filter=('xpath-filter', expr), 
                                withDefaults=withDefaults,
                                xPathNamespacePrefixMap=xPathNamespacePrefixMap, **kwargs)    
    return result

def _get_rpc_helper(m, expr, withDefaults="", xpath=False, xPathNamespacePrefixMap=None, **kwargs):
    """
    m: ncclient manager instance
    withDefaults: explicit, report-all, trim, report-all-tagged
    """
    #if withDefaults not in ['', 'explicit', 'report-all', 'trim', 'report-all-tagged']:
    #    raise 
    result=""
    if xpath == False:
        if expr == None:    
            #no filter
            result=m.get(withDefaults=withDefaults, **kwargs)                          
        else:   
            # expr passed in may be bytes or str so we use duct tape here
            try:
                expr = expr.decode("utf-8")
            except AttributeError:
                pass

            if expr.strip().startswith("<filter"):
                #already a filter str that includes <filter> element
                result=m.get(filter=expr, withDefaults=withDefaults, **kwargs)
            else:
                result=m.get(filter=('subtree', expr), withDefaults=withDefaults, **kwargs)
    else:
        if expr == None:      
                result=m.get(withDefaults=withDefaults, **kwargs)                
        else:   
                result=m.get(filter=('xpath', expr), 
                                withDefaults=withDefaults,
                                xPathNamespacePrefixMap=xPathNamespacePrefixMap, **kwargs)    
    return result
         
def _get_config_rpc_helper(m, source, expr, withDefaults="", xpath=False, xPathNamespacePrefixMap=None, **kwargs):
    """
    m: ncclient manager session
    withDefaults: explicit, report-all, trim, report-all-tagged
    """
    result=""
    if xpath == False:
        if expr == None:                              
            result=m.get_config(source,  withDefaults=withDefaults, **kwargs)                
        else:   
            #already a filter str that includes <filter> element
            # expr passed in may be bytes or str so we use duct tape here
            try:
                expr = expr.decode("utf-8")
            except AttributeError:
                pass
            if expr.strip().startswith("<filter"):
                result=m.get_config(source, filter=expr, withDefaults=withDefaults, **kwargs)
            else:    
                result=m.get_config(source, filter=('subtree', expr), withDefaults=withDefaults, **kwargs)
    else:    
        if expr == None:      
            result=m.get_config(source, withDefaults=withDefaults, **kwargs)                
        else:   
            result=m.get_config(source, filter=('xpath', expr), 
                            withDefaults=withDefaults,
                            xPathNamespacePrefixMap=xPathNamespacePrefixMap,
                            **kwargs)                   
    return result
    

def _query_yes_no(question, default="yes"):
    """Ask a yes/no question via raw_input() and return their answer.

    "question" is a string that is presented to the user.
    "default" is the presumed answer if the user just hits <Enter>.
        It must be "yes" (the default), "no" or None (meaning
        an answer is required of the user).

    The "answer" return value is one of "yes" or "no".
    """
    valid = {"yes":True,   "y":True,  "ye":True,
             "no":False,     "n":False}
    if default == None:
        prompt = " [y/n] "
    elif default == "yes":
        prompt = " [Y/n] "
    elif default == "no":
        prompt = " [y/N] "
    else:
        raise ValueError("invalid default answer: '%s'" % default)

    while True:
        sys.stdout.write(question + prompt)
        choice = input().lower()
        if default is not None and choice == '':
            return valid[default]
        elif choice in valid:
            return valid[choice]
        else:
            sys.stdout.write("Please respond with 'yes' or 'no' " "(or 'y' or 'n').\n")

def _processNodeIDValArg(arg):
    nodeIDVal=[]
    if isinstance(arg, str):
        #for get etc.
        arglist = arg.split("/")[1:]
        val=""
    else:
        #tuple, for edit etc.
        nid = arg[0]
        if len(arg)> 1:                
            val= arg[1]
        else:
            val = ""    
        arglist = nid.split("/")[1:]
    DEBUGPRINT("\targlist ", arglist)
    for i, nodeName in enumerate(arglist):
        DEBUGPRINT("\t process nodename in arglist ", nodeName)
        if i == len(arglist) -1 and nodeName == arglist[-1]:
            # set the last node in the path to val
            nameValtup=(nodeName, val)
        else:
            # set all intermediate node value in the path to empty
            nameValtup=(nodeName, '')
        nodeIDVal.append(nameValtup)                    
    return nodeIDVal

def _build_nodePathAndValueList(args):
    """
    # the returned nodeIDValueList is in the form 
    # [[(netconf-state, ""), (datastores, ""), ("datastore","")], [(),()...]], or
    # [[(netconf-state, ""), (datastores, ""), ("datastore","running")], [(),()...]]        
    """
    nodeIDValueList=[] 
    for arg in args:
        DEBUGPRINT("\targ here ", arg)
        nodeIDVal = _processNodeIDValArg(arg)
        DEBUGPRINT("\tnodeIDVal ", nodeIDVal)
        nodeIDValueList.append(nodeIDVal)
    return nodeIDValueList

def _build_getssubtree_expr(nodeIDValueList, prefix_namespace_dict=None, getdataFilter=False):
    """
    used internally. End users should use build_subtree_filter instead.
    nodeIDValueList: 
    """    
    topLevelNodesList=[]
    for nodeIDValueItem in nodeIDValueList:
        #print("_build_getssubtree_expr nodeIDValueItem ", nodeIDValueItem)
        _build_subtree_toplevel_elem_list_item(topLevelNodesList, nodeIDValueItem, prefix_namespace_dict=prefix_namespace_dict)
    if getdataFilter:
        filterNode= ElementTree.Element("subtree-filter")
    else:    
        filterNode= ElementTree.Element("filter", type="subtree")

    if topLevelNodesList is not None:
        for f in topLevelNodesList:
            filterNode.append(f)
    filterExpr = ElementTree.tostring(filterNode, encoding="utf-8", pretty_print=False)    
    return filterExpr

def _extract_value_op_wd_from_dict(v):
    #v is a tuple (e.g.): {'value':value, 'operation':'merge', 'with-default':True, 'insert-operation':"before",'target-key':(), 'target-value':tvalue })
    if "value" in v:
        value=v["value"]    
    else:
        value=""
        
    if "operation" in v:
        op=v["operation"]        
    else:
        op=""
        
    if "with_default" in v and v["with_default"] ==True:
        wd="with-default"        
    else:
        wd=""

    if "insert_operation" in v:
        insert_operation=v["insert_operation"]
    else:
        insert_operation=""
        
    if "target_key" in v:
        targetkey=v["target_key"]
    else:
        targetkey=""
        
    if "target_value" in v:
        targetvalue=v["target_value"]
    else:
        targetvalue     =""
        
    return (value, op, wd, insert_operation, targetkey,targetvalue)    
    
    
def _compute_xpath_identityref_value_nsmap(val, contextTopLevelNode, prefix_namespace_dict):
    # val can be an instance identifier or XPATH value,
    # such as in a partial lock rpc's select element:
    # <pl:select>/if:interfaces</pl:select>
    # <pl:select>/nacm:nacm</pl:select>            
    # in these cases
    # we must add the corresponding namespace to the branch's top contextTopLevelNode node
    #print "in coming val ", val
    nsmap_toadd = {}
    valnew = ""      
    if val == "":
        return (nsmap_toadd, val)
    
    if contextTopLevelNode is None or val.strip() == "":
        return (nsmap_toadd, val)
    
    vallist = val.split("/")      
    for eachval in vallist:
        if eachval =="":
            valnew = valnew + "/"
            continue
        idx = eachval.find(":")
        if idx != -1:                    
            eachvalPfx = eachval[0:idx]
            eachVal = eachval[idx+1:]
            #print "eachvalPfx ", eachvalPfx ,  " eachVal ",eachVal
            # we must exclude default namespace in use for contextTopLevelNode 
            # work around a duplicated xmlns:ncm (e.g.)
            # see sample-scripts/send_any_rpc.py
            if eachvalPfx in prefix_namespace_dict:
                # 
                #if eachvalPfx not in contextTopLevelNode.nsmap:
                # the check above is incorrect, the mapping does not have
                # to in a nodes' immediate parent (Context node)
                nsmap_toadd[eachvalPfx] = prefix_namespace_dict[eachvalPfx]
                valnew = valnew + eachval +"/"
                        
            elif eachvalPfx in SS_YANG_Dict:
                # dateAndTime value also contains : so we can not raise here
                #raise Exception("prefix '%s' used must be registered!"%eachvalPfx)    
                
                # if this is an identity value defined in another module modulename:identity, 
                # then we need to add namespace mapping
                #myprx = eachvalPfx
                myns = SS_YANG_Dict[eachvalPfx]["namespace"]
                
                for pfx, ns in list(prefix_namespace_dict.items()):
                    if ns == myns:
                        # there is already a prefix=>namespace mapping existed
                        # so we just use it
                        nsmap_toadd[pfx] = prefix_namespace_dict[pfx]
                        valnew = valnew + pfx+":"+eachVal + "/"                            
                        break
                else:      
                    # the following is only done when for loop exit normally                              
                    nsmap_toadd[eachvalPfx] = SS_YANG_Dict[eachvalPfx]["namespace"]
                    valnew = valnew  + eachval + "/"
            else:
                valnew = valnew + eachval + "/"
                                                  
        else:            
            valnew = valnew + eachval + "/"
    #print "out going val ", valnew[0:-1]            
         
    return (nsmap_toadd, valnew[0:-1])  

# All get and edit rpc ends up using this procedure
def _build_subtree_toplevel_elem_list_item_helper(eleItem, elementName, contextTopLevelNode, prefix_namespace_dict):
    val=""
    if type(eleItem[1]) is dict:
        # API uses this
        #print "elemItem", eleItem
        (val, op, wd, insert_operation, targetkey, targetvalue) = _extract_value_op_wd_from_dict(eleItem[1])
        #print "val , op, wd, ...", val, op, wd, insert_operation, targetkey, targetvalue
        # change modulename:value to prefix:value if the nsmap retunred 
        # uses a different prefix
        
        nsmap_toadd, val = _compute_xpath_identityref_value_nsmap(val, contextTopLevelNode, prefix_namespace_dict)
        # child topLevel Node           
        cldtopLevelNode=ElementTree.Element("%s"%elementName, nsmap=nsmap_toadd)
        
        if op !="":
            cldtopLevelNode.set(nsQualifierBase + "operation", op)
        if wd == "with-default": 
            cldtopLevelNode.set(nsQualifieWithDefaults + "default", "true")                           
        if insert_operation !="":
            cldtopLevelNode.set(nsQualifierYang + "insert", insert_operation) 
        #print("targetkey ", targetkey)                
        if targetkey != "":        
            if not isinstance(targetkey, str):
                # this is passed in as a list of (keyName, value) tuple
                convertedTargetKey = ""
                nodeIDPath = eleItem[0].strip("/") # strip fist "/" so we don't get the first item as "" when splitting by "/"
                fistNodeName = nodeIDPath.split("/")[0] 
                
                namespace = ElementTree.QName(elementName).namespace
                keyPfx = ""
                
                if fistNodeName.find(":") != -1:
                    keyPfx = fistNodeName.split(":")[0] 
                    # in case module-name is used, map it to the known prefix value
                    if keyPfx not in prefix_namespace_dict:
                        # module identifier is used in this case
                        if keyPfx in SS_YANG_Dict:
                            keyPfx = SS_YANG_Dict[keyPfx]["prefix"]
                          
                for keyName, value in targetkey:                    
                    if keyName.find(":") == -1:
                        # check if a prefix is used in the pathID
                        convertedTargetKey = convertedTargetKey +  "[%s:%s='%s']"%(keyPfx, keyName, value)
                    else:
                        keyPrefixSpecified, keyNameSpecified = keyName.split(":")
                        if keyPrefixSpecified != keyPfx:
                            # keyPrefixSpecified must be a module name so we should not use it
                            convertedTargetKey = convertedTargetKey +  "[%s:%s='%s']"%(keyPfx, keyNameSpecified, value)
                        else:
                            # a prefix is already specified
                            convertedTargetKey = convertedTargetKey +  "[%s='%s']"%(keyName, value)
                                 
                targetkey = convertedTargetKey 
            
            cldtopLevelNode.set(nsQualifierYang + "key", targetkey)  
            
        if targetvalue != "":       
            cldtopLevelNode.set(nsQualifierYang + "value", targetvalue)    
        #print "val in a dict ", val    
        if val != "":
            if "raw_XML_data" in eleItem[1] and eleItem[1]["raw_XML_data"]:
                # val musr be an xml block
                anyXMLNode= ElementTree.fromstring(val)
                cldtopLevelNode.append(anyXMLNode)
            else:                                 
                cldtopLevelNode.text=val       
        
    else:
        val=eleItem[1]
        #print("val not a dict ", val, "  elementName ", elementName)
        nsmap_toadd, val = _compute_xpath_identityref_value_nsmap(val, contextTopLevelNode, prefix_namespace_dict)
        #print("nsmap_toadd " ,  nsmap_toadd, " val ", val )
        # child topLevel Node           
        cldtopLevelNode=ElementTree.Element("%s"%elementName, nsmap=nsmap_toadd)              
        if val == "INSERT_ATTRIBUTE_TEST_SEGUESOFT":
            cldtopLevelNode.attrib["fakeTestAttr"] = "segueTest"                          
        elif val != "":  
            cldtopLevelNode.text=val
            
            
    return cldtopLevelNode
        
def _build_subtree_toplevel_elem_list_item(topLevelNodesList, nodeIDValueItem, 
                indentflag=True, prefix_namespace_dict=None,
                keepInputNode=False):
    """     
    # value passed in as dictionary (used by API)
    # path root to leaf:  [('interfaces', ''), ('interface', ''), ('name', {'value':'ggg', 'raw_XML_data': True', operation':'replace', 'insert_op':'before', 'taget_key':tkey, 'target_value:tval, 'with_default':True) )]
    # path root to leaf:  [('interfaces', ''), ('interface', ''), ('counters', ''), ('inBytes', (u'nn', ''))]    
    """
    # 
    # [('nacm:nacm', ''), ('nacm:rule-list', {'raw_XML_data': False, 'value': ''})], 
    # [('nacm:nacm', ''), ('nacm:rule-list', ''), ('nacm:name', {'raw_XML_data': False, 'value': 'a'})], 
                
    topLevelNode = None 
    eleItem = nodeIDValueItem[0]
    elementName = eleItem[0]
    
    newTopLevelListEntry =False        
    if not isinstance(eleItem[1], str):
        if "new_list_entry" in eleItem[1]:         
            newTopLevelListEntry = True
            
    # must check the following pathID ( i.e., possible the first list key node)    
    # why only the second pathID?    Because we are only consdering the first node here!
    if len(nodeIDValueItem) > 1:       
        if not isinstance(nodeIDValueItem[1], str):                
            if "new_list_entry" in eleItem[1]:         
                newTopLevelListEntry = True              
                          
    #print("elementName ", elementName, " prefix_namespace_dict " , prefix_namespace_dict)
    elementName = _construct_namespace_qualified_name(elementName, prefix_namespace_dict)
    # it is important we always use the toplevel node that is created most recently
    # for example, create multiple list entries for a top level list (see below example in 
    # comments)
    for n in topLevelNodesList[::-1]:
        # same root, existed already, just use it
        if n.tag == elementName and not newTopLevelListEntry:
            topLevelNode = n
            break
    
    # this top level node has not been created yet so go to create it   
    # in case the toplevel node is a list, for example 
    #        [('b:server', {'raw_XML_data': False, 'value': ''})], 
    #        [('b:server', ''), ('b:name', {'raw_XML_data': False, 'value': 'a'})], 
    #        [('b:server', {'raw_XML_data': False, 'value': ''})], 
    #        [('b:server', ''), ('b:name', {'raw_XML_data': False, 'value': 'b'})]                
    # in this case eleItem == nodeIDValueItem[-1] ==>  if the elemItem is the
    # last node name (not any intermediate ones leading to the node name
    # that is explicitly called out by the caller and always need to be created    
    # 
    # again we must check the length here because we are only consdering the first node here!
    if topLevelNode is None or  (eleItem == nodeIDValueItem[-1] and len(nodeIDValueItem) == 1):
        topLevelNode = _build_subtree_toplevel_elem_list_item_helper(eleItem, elementName, None, prefix_namespace_dict)
        topLevelNodesList.append(topLevelNode)   

    #print "topLevelNodesList ", topLevelNodesList
    # attach the remaining child nodes in this path
    # note in YANG module, node name is not required to be unique, except the top level
    # so we must (xpath) search an existing node using its parent as context
    contextTopLevelNode = topLevelNode
    nodeIDPathsExceptFirstOne = nodeIDValueItem[1:]
    for i, eleItem in enumerate(nodeIDPathsExceptFirstOne):      
        #eleItem (value ) can be a tuple that contains attribute values        
        elementName = eleItem[0]
        if elementName.endswith(":input") and keepInputNode ==False:
            # if input is given in the ID path when calling send_any_action, ignore it
            continue
        newListEntry = False
        if not isinstance(eleItem[1], str):                
            if "new_list_entry" in eleItem[1]:         
                newListEntry = True          
                
        # must check the next pathID ( i.e., possibly the first list key node when the
        # list node isn't specified)        
        if len(nodeIDPathsExceptFirstOne) > i+1:       
            if not isinstance(nodeIDPathsExceptFirstOne[i+1], str):                
                if "new_list_entry" in nodeIDPathsExceptFirstOne[i+1][1]:         
                    newListEntry = True          
                
        # check if the name includes prefix:
        #idx = elementName.find(":")    
        
        elementName = _construct_namespace_qualified_name(elementName, prefix_namespace_dict)
        #print " qualified name %s", elementName
        existedIntermediateNode = None
        existedIntermediateNodes = contextTopLevelNode.findall("./%s"%elementName)
        if len(existedIntermediateNodes) > 0:
            # use the last one from the nodes found. That should be the *last one*
            # just added in the previous iteration            
            existedIntermediateNode = existedIntermediateNodes[-1]
        
        existed=False
        if existedIntermediateNode is not None:
            existed = True                    
        
        # Modifying the namespace mapping of a node is not possible in lxml. 
        # See https://bugs.launchpad.net/lxml/+bug/555602
        # open ticket that has this feature as a wishlist item.
        
        # so before we create cldtopLevelNode we need to find all nsmaps needed
        #
        # in case eleItem == nodeIDValueItem[-1] ==>  if the elemItem is the
        # last node name (not any intermediate ones leading to the node name
        # that is explicitly called out by the caller and always need to be created
        #print "eleItem ", eleItem, " -- eleItem == nodeIDValueItem[-1]) ?", eleItem == nodeIDValueItem[-1]
        # for eleItem == nodeIDValueItem[-1]), since this must be a leaf so we will always need to create it
        if existed == False or newListEntry or (existed == True and eleItem == nodeIDValueItem[-1]):            
            cldtopLevelNode = _build_subtree_toplevel_elem_list_item_helper(eleItem, elementName, contextTopLevelNode, prefix_namespace_dict )
            contextTopLevelNode.append(cldtopLevelNode)
            #change contextTopLevelNode to newly created one
            contextTopLevelNode = cldtopLevelNode
        else:
            contextTopLevelNode = existedIntermediateNode        
    return topLevelNodesList

        
class NamespaceMissingError(Exception):
    def __init__(self, prefix):
        self.missing_preifx =prefix
         
    def __str__(self):
        return "The namespace referenced by prefix:'%s' was not found. Must call register_namespace to register it first!"%self.missing_preifx

class ParameterError(Exception):
    def __init__(self, error):
        self.parameter_error =error
         
    def __str__(self):
        return "%s"%self.parameter_error
