#---------------------------------------------------------------------
# Copyright (C) 2013  Seguesoft  Inc.
#                                                                             
# Redistribution of this software, in whole or in part, is prohibited         
# without the express written permission of Seguesoft. 
# Modified based on ncclient
# ----------------------------------------------------------------------
# Copyright 2009 Shikhar Bhushan
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""This module is a thin layer of abstraction around the library. It exposes all core functionality."""

from . import capabilities
from . import msgIDgen
from . import operations
from . import transport

import logging, six

logger = logging.getLogger('ncclient.manager')

CAPABILITIES = [
    "urn:ietf:params:netconf:base:1.0",
    "urn:ietf:params:netconf:base:1.1",
    "urn:ietf:params:netconf:capability:writable-running:1.0",
    "urn:ietf:params:netconf:capability:candidate:1.0",
    "urn:ietf:params:netconf:capability:confirmed-commit:1.0",
    "urn:ietf:params:netconf:capability:confirmed-commit:1.1",
    "urn:ietf:params:netconf:capability:rollback-on-error:1.0",
    "urn:ietf:params:netconf:capability:startup:1.0",
    "urn:ietf:params:netconf:capability:url:1.0?scheme=http,ftp,file,https,sftp",
    "urn:ietf:params:netconf:capability:validate:1.0",
    "urn:ietf:params:netconf:capability:validate:1.1",
    "urn:ietf:params:netconf:capability:xpath:1.0",    
    "urn:ietf:params:xml:ns:netconf:notification:1.0",
    "urn:ietf:params:netconf:capability:interleave:1.0",
    "urn:ietf:params:netconf:capability:with-defaults:1.0?basic-mode=explicit&also-supported=trim,report-all,report-all-tagged",
]
"""A list of URI's representing the client's capabilities. This is used during the initial capability exchange. Modify this if you need to announce some capability not already included."""

OPERATIONS = {
    "get": operations.Get,
    "get_data": operations.GetData,    
    "get_config": operations.GetConfig,
    "edit_config": operations.EditConfig,
    "edit_data": operations.EditData,
    "copy_config": operations.CopyConfig,
    "validate": operations.Validate,
    "commit": operations.Commit,
    "discard_changes": operations.DiscardChanges,
    "delete_config": operations.DeleteConfig,
    "lock": operations.Lock,
    "unlock": operations.Unlock,
    "close_session": operations.CloseSession,
    "kill_session": operations.KillSession,
    "poweroff_machine": operations.PoweroffMachine,
    "reboot_machine": operations.RebootMachine,
    "send_xml": operations.SendXML,
    "create_subscription":operations.CreateSubscription,
    "cancel_commit":operations.CancelCommit,
    "discard_changes":operations.DiscardChanges,
    "get_schema":operations.GetSchema
}
"""Dictionary of method names and corresponding :class:`~ncclient.operations.RPC` subclasses. It is used to lookup operations, e.g. `get_config` is mapped to :class:`~ncclient.operations.GetConfig`. It is thus possible to add additional operations to the :class:`Manager` API."""

def connect_ssh(*args, **kwds):
    """Initialize a :class:`Manager` over the SSH transport. For documentation of arguments see :meth:`ncclient.transport.SSHSession.connect`.

    The underlying :class:`ncclient.transport.SSHSession` is created with :data:`CAPABILITIES`. It is first instructed to :meth:`~ncclient.transport.SSHSession.load_known_hosts` and then  all the provided arguments are passed directly to its implementation of :meth:`~ncclient.transport.SSHSession.connect`.
    """
    
    if "timeout" in kwds:
        timeout=kwds["timeout"]
    else:
        timeout =None
        
    if timeout is None:
        timeout=60
        
        
    if "capability" in kwds:
        if kwds["capability"] is not None:
            #CAPABILITIES.extend(kwds["capability"])
            #capability = CAPABILITIES
            if "EXTEND_CAPABILITY_FLAG_SET" in kwds["capability"]:                                
                capability = CAPABILITIES[:]
                capability.extend(kwds["capability"])
                capability.remove("EXTEND_CAPABILITY_FLAG_SET")   
            elif len(kwds["capability"]) == 1 and kwds["capability"][0] == "DO_NOT_SEND_1.0_CAPABILITY":
                capability= kwds["capability"]
                capability.extend(CAPABILITIES)          
            elif len(kwds["capability"]) == 1 and kwds["capability"][0] == "DO_NOT_SEND_1.1_CAPABILITY":
                capability= kwds["capability"]
                capability.extend(CAPABILITIES)                                    
                          
            else:
                capability =  kwds["capability"]    
        else:
            capability = CAPABILITIES
            
            
        if "DO_NOT_SEND_1.0_CAPABILITY" in capability:  
            capability.remove("DO_NOT_SEND_1.0_CAPABILITY")
            if "urn:ietf:params:netconf:base:1.0" in capability:
                capability.remove("urn:ietf:params:netconf:base:1.0")

        if "DO_NOT_SEND_1.1_CAPABILITY" in capability:  
            capability.remove("DO_NOT_SEND_1.1_CAPABILITY")
            if "urn:ietf:params:netconf:base:1.1" in capability:
                capability.remove("urn:ietf:params:netconf:base:1.1")
                        
        #must remove it 
        del kwds["capability"]
    else:
        capability = CAPABILITIES
        
        
    session = transport.SSHSession(capabilities.Capabilities(capability),timeout=timeout)
    session.load_known_hosts()
    session.connect(*args, **kwds)
    return Manager(session)

connect = connect_ssh
"Same as :func:`connect_ssh`"

def connect_tls(*args, **kwds):
    """Initialize a :class:`Manager` over the TLS transport. `.
    """
    
    if "timeout" in kwds:
        timeout=kwds["timeout"]
    else:
        timeout =None
        
    if timeout is None:
        timeout=60
        
        
    if "capability" in kwds:
        if kwds["capability"] is not None:
            #CAPABILITIES.extend(kwds["capability"])
            #capability = CAPABILITIES
            if "EXTEND_CAPABILITY_FLAG_SET" in kwds["capability"]:                                
                capability = CAPABILITIES[:]
                capability.extend(kwds["capability"])
                capability.remove("EXTEND_CAPABILITY_FLAG_SET")
            elif len(kwds["capability"]) == 1 and kwds["capability"][0] == "DO_NOT_SEND_1.0_CAPABILITY":
                capability= kwds["capability"]
                capability.extend(CAPABILITIES) 
            elif len(kwds["capability"]) == 1 and kwds["capability"][0] == "DO_NOT_SEND_1.1_CAPABILITY":
                capability= kwds["capability"]
                capability.extend(CAPABILITIES)                                    
            else:
                capability =  kwds["capability"]    
        else:
            capability = CAPABILITIES
            
        if "DO_NOT_SEND_1.0_CAPABILITY" in capability:  
            capability.remove("DO_NOT_SEND_1.0_CAPABILITY")
            if "urn:ietf:params:netconf:base:1.0" in capability:
                capability.remove("urn:ietf:params:netconf:base:1.0")     
                
        if "DO_NOT_SEND_1.1_CAPABILITY" in capability:  
            capability.remove("DO_NOT_SEND_1.1_CAPABILITY")
            if "urn:ietf:params:netconf:base:1.1" in capability:
                capability.remove("urn:ietf:params:netconf:base:1.1")
                       
        #must remove it 
        del kwds["capability"]
    else:
        capability = CAPABILITIES
        
        
    session = transport.TLSSession(capabilities.Capabilities(capability),timeout=timeout)
    session.connect(*args, **kwds)
    return Manager(session)

class OpExecutor(type):

    def __new__(cls, name, bases, attrs):
        def make_wrapper(op_cls):
            def wrapper(self, *args, **kwds):
                return self.execute(op_cls, *args, **kwds)
            wrapper.__doc__ = op_cls.request.__doc__
            return wrapper
        for op_name, op_cls in OPERATIONS.items():
            attrs[op_name] = make_wrapper(op_cls)
        return super(OpExecutor, cls).__new__(cls, name, bases, attrs)


#class Manager(object, metaclass=OpExecutor):
class Manager(six.with_metaclass(OpExecutor)):
    """For details on the expected behavior of the operations and their parameters refer to :rfc:`4741`.

    Manager instances are also context managers so you can use it like this::

        with manager.connect("host") as m:
            # do your stuff

    ... or like this::

        m = manager.connect("host")
        try:
            # do your stuff
        finally:
            m.close_session()
    """

    def __init__(self, session, timeout=30):
        self._session = session
        self._async_mode = False
        self._timeout = timeout
        self._raise_mode = operations.RaiseMode.NONE
        self._msgID_gen = msgIDgen.MsgIDGen()
        
    def __enter__(self):
        return self

    def __exit__(self, *args):
        self.close_session()
        return False

    def __set_timeout(self, timeout):
        self._timeout = timeout

    def __set_async_mode(self, mode):
        self._async_mode = mode

    def __set_raise_mode(self, mode):
        assert(mode in (operations.RaiseMode.NONE, operations.RaiseMode.ERRORS, operations.RaiseMode.ALL))
        self._raise_mode = mode

    def __set_delim_ver(self, ver):
        self._delim_ver = ver
        
    def execute(self, cls, *args, **kwds):
        try:
            return cls(self._session,
                   asyncOp=self._async_mode,
                   timeout=self._timeout,
                   raise_mode=self._raise_mode,
                   msgID_gen=self._msgID_gen).request(*args, **kwds)
        except Exception as e:
            try:
                import wx
                try:
                    wx.EndBusyCursor()
                except:
                    pass
                    
                # THIS dialog is not needed and would cause crash 
                # on Linux GTK2 if invoked from a thread that is
                # not the main UI thread.
                #dlg=wx.MessageDialog(None, repr(e), "Execute Error",
                #            wx.OK | wx.ICON_ERROR)    
                #dlg.ShowModal()
                #dlg.Destroy()
                raise e
            except:            
                raise e
            
    def locked(self, target):
        """Returns a context manager for a lock on a datastore, where *target* is the name of the configuration datastore to lock, e.g.::

            with m.locked("running"):
                # do your stuff

        ... instead of::

            m.lock("running")
            try:
                # do your stuff
            finally:
                m.unlock("running")
        """
        return operations.LockContext(self._session, target)

    #def listen_notification(self, callback):
    #    self._session.listen_notification(callback)
    #@property
    #def sshsession(self):
    #    "SSHSession of this netconf session."
    #    return self._session
        
    @property
    def session(self):
        "SSHSession or TLSSession of this netconf session."
        return self._session

    @property
    def client_capabilities(self):
        ":class:`~ncclient.capabilities.Capabilities` object representing the client's capabilities."
        return self._session._client_capabilities

    @property
    def server_capabilities(self):
        ":class:`~ncclient.capabilities.Capabilities` object representing the server's capabilities."
        return self._session._server_capabilities

    @property
    def session_id(self):
        "`session-id` assigned by the NETCONF server."
        return self._session.id

    @property
    def session_ciphers(self):
        "`session-ciphers` supported by ssl session"
        return self._session.ciphers

    @property
    def connected(self):
        "Whether currently connected to the NETCONF server."
        return self._session.connected

    @property
    def host(self):
        return self._session._host
        
    async_mode = property(fget=lambda self: self._async_mode, fset=__set_async_mode)
    "Specify whether operations are executed asynchronously (`True`) or synchronously (`False`) (the default)."

    timeout = property(fget=lambda self: self._timeout, fset=__set_timeout)
    "Specify the timeout for synchronous RPC requests."

    raise_mode = property(fget=lambda self: self._raise_mode, fset=__set_raise_mode)
    "Specify which errors are raised as :exc:`~ncclient.operations.RPCError` exceptions. Valid values are the constants defined in :class:`~ncclient.operations.RaiseMode`. The default value is :attr:`~ncclient.operations.RaiseMode.ALL`."
