# Copyright (C) Seguesoft                                               
#                                                                             
# Redistribution of this software, in whole or in part, is
# prohibited without the express written permission of 
# Seguesoft. 
#
import collections, re
from lxml import etree
import parse_module
import ssyangtree
from common_global_variables import _dataNodesNoValue
from common_global_variables import DEBUGPRINT
   
class KeyOrderError(Exception):
   """Raised when KeyOrderError"""
   pass   
    
def getAndReturnInstNodeYANGInstanceID(ssession, instElemNode):
    retNSmap = {}
    retNodeIDAndValues = []   
    getInstNodeYangInstanceID(
            instElemNode,  
            instElemNode.getroottree().getroot(),
            retNodeIDAndValues, 
            retNSmap)
 
    return (retNodeIDAndValues, retNSmap)
    
def getInstNodeYangInstanceID(
            elemNode, elemRootNode, 
            retNodeIDAndValues,
            retNSmap, 
            session= None,
            sameInstanceListEntryFlagDict={"flag":True}, 
            separateLeafListIID=False):       
    '''The following are examples of instance identifiers:

     /* instance-identifier for a container */
     /ex:system/ex:services/ex:ssh

     /* instance-identifier for a leaf */
     /ex:system/ex:services/ex:ssh/ex:port

     /* instance-identifier for a list entry */
     /ex:system/ex:user[ex:name='fred']

     /* instance-identifier for a leaf in a list entry */
     /ex:system/ex:user[ex:name='fred']/ex:type

     /* instance-identifier for a list entry with two keys */
     /ex:system/ex:server[ex:ip='192.0.2.1'][ex:port='80']

     /* instance-identifier for a leaf-list entry */
     /ex:system/ex:services/ex:ssh/ex:cipher[.='blowfish-cbc']
     But we usually  use the list of value to indicate a leaflist
     /ex:system/ex:services/ex:ssh/ex:cipher => [val1, val2, etc.]
    
     no key list won't appear in the config, only state data
     /* instance-identifier for a list entry without keys */
     /ex:stats/ex:port[3]
    '''
    DEBUGPRINT("getInstNodeYangInstanceID ", elemNode)

    # Check if there is any 'key' node in this node's ancestors
    # if yes, we must include them in the set
    keyNodeVisited = set()
    keyNodeIDAndValues = []  
   
    includeKeyNodeAndKeyNodesInAncestors(
        elemNode,  
        elemRootNode,
        keyNodeIDAndValues,
        retNSmap, 
        keyNodeVisited, 
        sameInstanceListEntryFlagDict=sameInstanceListEntryFlagDict,
        includeSelectedNodeSelfToUpdate=False )

    DEBUGPRINT("includeKeyNodeAndKeyNodesInAncestors got :%s"% keyNodeIDAndValues)
    
    yangTreeRootNode = getYangTreeRoot(elemNode, elemRootNode)
    DEBUGPRINT("found yangTreeRootNode ", parse_module.getArgumentValue(yangTreeRootNode))
    
    yangElemNode = getInstanceNodeYangElem(elemNode, elemRootNode)
    DEBUGPRINT("# found yangElemNode ", yangElemNode)

    if yangElemNode is None:
        raise Exception("The YANG element that defines the instance data "\
                    " %s not found!"%elemNode.tag)  

    selectedNodeIDValueDict = None
    if yangTreeRootNode is not None:
        selectedNodeIDValueDict = getInstElemNodeIDAndValueDict(
                        yangTreeRootNode,   
                        elemNode,
                        yangElemNode)
    #DEBUGPRINT("selectedNodeIDValueDict ", selectedNodeIDValueDict)
    if selectedNodeIDValueDict is not None:
        (nodeIDAndValue, mymap) = selectedNodeIDValueDict
        
        # convert nodeIDAndValue to IID, based on the 
        # keyNodeIDAndValues discovered
        # e.g. keyNodeIDAndValues [
        #    ('/nacm:nacm/nacm:rule-list', {'new_list_entry': True, 'value': ''}), 
        #    ('/nacm:nacm/nacm:rule-list/nacm:name', {'value': 'almighty'}),
        #    ('/nacm:nacm/nacm:rule-list/nacm:rule', {'value': ''}), 
        #    ('/nacm:nacm/nacm:rule-list/nacm:rule/nacm:name', {'value': 'almighty'})]
        #
        #     nodeIDAndValue ('/nacm:nacm/nacm:rule-list/nacm:rule/nacm:action',
        #                            {'value': 'permit'})
        # partial lock got nodeIDs 
        #     ('/nacm:nacm/nacm:rule-list/nacm:rule/nacm:action', {'value': 'permit'})
        # convert to
        #     ('/nacm:nacm/nacm:rule-list[nacm:name='almighty']/nacm:rule[nacm:name='almight']/nacm:action',
        #                                                {'value': 'permit'})
        #DEBUGPRINT("instance nodeIDAndValue ", nodeIDAndValue, " orignalNodeID ", nodeIDAndValue[0])
        orignalNodeID = nodeIDAndValue[0]        
        insertedIndexDict = collections.OrderedDict()
                            
        if len(keyNodeIDAndValues) == 0:
            # TO do: we need to handle not key list entry!
            if nodeIDAndValue[1]["leaflist_node"]:
                if separateLeafListIID: 
                #  /* instance-identifier for a leaf-list entry */
                #  /ex:system/ex:services/ex:ssh/ex:cipher[.='blowfish-cbc']
                    convertedNodeIDAndValue = ("%s[.='%s']"%(nodeIDAndValue[0], 
                            nodeIDAndValue[1]["value"]), nodeIDAndValue[1])
                else:
                    convertedNodeIDAndValue = (nodeIDAndValue[0],
                                               nodeIDAndValue[1])                             
            else:
                convertedNodeIDAndValue = nodeIDAndValue[:]                         
        else:       
            for keyNodeIDValue in keyNodeIDAndValues:
                #DEBUGPRINT("keyNodeIDValue ", keyNodeIDValue)
                nodePath = keyNodeIDValue[0]
                nodeValue = keyNodeIDValue[1]["value"]
                #print "handle nodePath ", nodePath
                                  
                if orignalNodeID.startswith(nodePath) and \
                    keyNodeIDValue[1]["list_node"] or \
                    orignalNodeID == nodePath and \
                    not keyNodeIDValue[1]["listkey_node"]:
                    # this must be the node itself or
                    # the list node returned. List node is also
                    # returned when calling                                 
                    continue
                else:            
                    # this must be a key node 
                    keyPartName = nodePath.split("/")[-1]
                    listNodePath = "/" .join(nodePath.split("/")[:-1])                                               
                    keyNameValueBracketPair = "[%s='%s']"%(keyPartName, nodeValue)
                    if listNodePath in insertedIndexDict:
                        insertedIndexDict[listNodePath] = \
                            insertedIndexDict[listNodePath] +\
                                 keyNameValueBracketPair
                    else:     
                        insertedIndexDict[listNodePath] =\
                            keyNameValueBracketPair
                 
            #construct IID
            insertItems = list(insertedIndexDict.items())
            insertItems.reverse()
            # we need to match from right side, starting 
            # from the deepest list node path 
            insertedIndexDictReversed = collections.OrderedDict(insertItems)   
            #                     
            nodeInstanceID = orignalNodeID
            #print "nodeInstanceID " , nodeInstanceID
            # if nodeInstanceID == 
            for listNodePath, insertedIndex in \
                    list(insertedIndexDictReversed.items()):
                if nodeInstanceID.startswith(listNodePath):  
                    nodeInstanceID = nodeInstanceID.\
                        replace(listNodePath, listNodePath +insertedIndex)
                      
            #  /* instance-identifier for a leaf-list entry */
            #  /ex:system/ex:services/ex:ssh/ex:cipher[.='blowfish-cbc']
            
            # to-do
            # if elemNode selected is a list key, the last part of
            # IID may need to be removed
            # i.e.  /ietf-netconf-monitoring:netconf-state/datastores/datastore[name='startup']/name
            # may need to be  /ietf-netconf-monitoring:netconf-state/datastores/datastore[name='startup']
            if nodeIDAndValue[1]["leaflist_node"]:
                if separateLeafListIID:
                    convertedNodeIDAndValue = ("%s[.='%s']"%(nodeInstanceID, 
                            nodeIDAndValue[1]["value"]), nodeIDAndValue[1])
                else:
                    convertedNodeIDAndValue = (nodeInstanceID, nodeIDAndValue[1])    
            else:                             
                convertedNodeIDAndValue = (nodeInstanceID, nodeIDAndValue[1])                     
        DEBUGPRINT("convertedNodeIDAndValue ", convertedNodeIDAndValue)        
        retNodeIDAndValues.append(convertedNodeIDAndValue)
        retNSmap.update(mymap)

def includeKeyNodeAndKeyNodesInAncestors( 
        instanceNode, 
        rootNode,
        nodeIDAndValuesToUpdate, 
        nsmapToUpdate,
        keyNodeVisited, 
        alreadyStagedNodeIDAndValues=[], 
        sameInstanceListEntryFlagDict={"flag":True},
        includeSelectedNodeSelfToUpdate= True):

    # 
    # includeSelectedNodeSelfToUpdate             
    # the select instance node itself
    # this is needed when calling includeKeyNodeAndKeyNodesInAncestors 
    # from instance_tree_list_model: getDirtyNodeYangIDs 

    # if a list is selected, we need to include its keys
    if _isList(instanceNode, rootNode):
        for child in instanceNode.iterchildren():
            if _isListKey(child, rootNode): 
                instanceNode = child

    ancestorsFromTop=[]        
    for ancestor in instanceNode.iterancestors():
        if ancestor.tag == "{urn:ietf:params:xml:ns:netconf:base:1.0}data":
            break
        ancestorsFromTop.append(ancestor)
    
    ancestorsFromTop.reverse()
    listEntryNotChecked = True    
        
    #new_list_entry_already_set_in_parent = False
    for ancestor in ancestorsFromTop:     
        if _isList(ancestor, rootNode):
            nodeIDsIncludeKeys = []
            # must first check if this has been visited
            # if we are doing "add staged values to next edit-config, 
            # we treat every node  none node as already visited
            # just like we are working on yang tree 
            if ancestor not in keyNodeVisited:   
                # print "include keys , process ancestor,
                # and mark it visited ", ancestor.tag,
                #         " value ", ancestor.text 
                keyNodeVisited.add(ancestor)
                # any chained sublist should be using this as parent 
                # list so it should be considered visited
                listEntryNotChecked = False
            else:
                continue
            
            # list key must be encoded first, and in the order give 
            # in 'key' statement
            # not in the order the statement nodes are actually defined, 
            # and before other nodes in set,
            # in get it is also the recommended way thought not required
            listKeyValue = _getListKey(ancestor, rootNode)
            keyLst = []
            if listKeyValue is not None:
                keyLst = listKeyValue.split()
        
            if len(keyLst) > 0:
                keyValueDict={}
                keyValueOrderedLst = []
                ListkeyUnReOrderedLst = []
                
                # add "new_list_entry" attribute is not needed 
                # since here we always specify the list entry explicitly
                IDsToSet = getInstElemNodeIDAndValueDict(
                        getYangTreeRoot(ancestor, rootNode),
                        ancestor,
                        getInstanceNodeYangElem(ancestor, rootNode ))
                if IDsToSet is not None:                    
                    (ancestorNodeIDAndValue, mymap) = IDsToSet
                    ########################################
                    nodeIDsIncludeKeys.append(ancestorNodeIDAndValue)    
                    nsmapToUpdate.update(mymap)
                                    
                for child in ancestor.iterchildren():
                    if _isListKey(child, rootNode): 
                        keyName = etree.QName(child).localname
                        IDsToSet = getInstElemNodeIDAndValueDict(\
                                getYangTreeRoot(child, rootNode), 
                                child,                                                               
                                getInstanceNodeYangElem(child, rootNode ))
                        
                        if IDsToSet is not None:
                            if child not in keyNodeVisited:    
                                keyNodeVisited.add(child)
                            else:
                                continue    
                            
                            (nodeIDAndValue, mymap) = IDsToSet
                            keyValueDict[keyName] = nodeIDAndValue
                            nsmapToUpdate.update(mymap)
                            ListkeyUnReOrderedLst.append(keyName)
           
                for key in keyLst:
                    if key in  keyValueDict:
                        keyValueOrderedLst.append(keyValueDict[key])
                #DEBUGPRINT("ListkeyUnReOrderedLst %s "% ListkeyUnReOrderedLst)

                # test using ietf-netconf-monitoring
                #if keyLst[0] == "identifier":
                #   keyLst=["identifier", "format", "version"] 
                if keyLst !=  ListkeyUnReOrderedLst and parse_module.is_running_cftest:
                    raise KeyOrderError("List key %s muse be encoded in the order "\
                                        " defined: %s"%(keyLst, ListkeyUnReOrderedLst))
                                    
                #DEBUGPRINT("keyValueUnReOrderedLst %s "% keyValueUnReOrderedLst)
                # check if all key nodes value sequence have already existed 
                # in the data assembled previously
                # by default we assume the same list entry
                # but once we detect the parent is not the same entry the 
                # its children are not neither
                    
                if sameInstanceListEntryFlagDict["flag"]:                            
                    sameInstanceListEntryFlagDict["flag"] = \
                        parse_module._checkIfSameInstanceSpecified(\
                            keyValueOrderedLst, alreadyStagedNodeIDAndValues)
                               
                if not sameInstanceListEntryFlagDict["flag"]:                             
                    nodeIDsIncludeKeys.extend(keyValueOrderedLst)
                else:
                    # get rid of the list entry itself
                    if ancestorNodeIDAndValue in nodeIDsIncludeKeys:
                        nodeIDsIncludeKeys.remove(ancestorNodeIDAndValue)                            
                nodeIDAndValuesToUpdate.extend(nodeIDsIncludeKeys)
                
    if includeSelectedNodeSelfToUpdate:                
        # the select instance node itself
        # this is needed when calling includeKeyNodeAndKeyNodesInAncestors from
        # instance_tree_list_model: getDirtyNodeYangIDs 
        yangTreeRoot = getYangTreeRoot(instanceNode,rootNode)
        if yangTreeRoot is None:    
            raise Exception("The YANG module file that defines data "\
                                " %s not found!"%instanceNode.tag)     
    
        dirtyNodeIDs = getInstElemNodeIDAndValueDict(yangTreeRoot, 
                        instanceNode,
                        getInstanceNodeYangElem(instanceNode, rootNode ))
        if dirtyNodeIDs is not None:
            (nodeIDAndValue, mymap) = dirtyNodeIDs
            #print "instanceNode not in keyNodeVisited ", instanceNode
            # it might be a key that is already visited
            leaflistNodeCat = parse_module.getStatementValue(\
                getInstanceNodeYangElem(instanceNode, rootNode ))
            if instanceNode not in keyNodeVisited and \
                    (not sameInstanceListEntryFlagDict["flag"] or \
                    listEntryNotChecked or leaflistNodeCat=="leaf-list"):
                nodeIDAndValuesToUpdate.append(nodeIDAndValue)
                nsmapToUpdate.update(mymap)
                keyNodeVisited.add(instanceNode)             

                           
def getInstanceNodeYangElem(node, rootNode):
    # rootNode: in the instance tree where this node is not a node 
    # in yang tree. It is actually <reply<, <data> etc.                
    foundTreeRoot = getYangTreeRoot(node, rootNode)   
    
    if foundTreeRoot is not None:
        DEBUGPRINT("\tgetInstanceNodeYangElem foundTreeRoot ", parse_module.getArgumentValue(foundTreeRoot))
        # rootNode is data root such as <data> element
        instanceNodeIDPath = getInstanceIDPath(node, rootNode)
        DEBUGPRINT("\tgetInstanceNodeYangElem instanceNodeIDPath: ", instanceNodeIDPath)            
        return parse_module.get_elemNode_from_IDPath(
                    instanceNodeIDPath, foundTreeRoot)
    
    raise Exception("No corresponding YANG tree found for "\
                            "the instance node %s"%node.tag)
       
def getInstElemNodeIDAndValueDict(foundTreeRoot, node, yangElemNode):
    (nodeID, mymap) = parse_module.getNodeIDPath(yangElemNode, 
                                    rootNode=foundTreeRoot)
    nodeValue = node.text
    if nodeValue is None or nodeValue.isspace():
        # the node.text obtained here can be pretty printed result like '\n          '
        # using that form will cause indenting to fail
        nodeValue = ""
                
    value =  {"value": nodeValue}   
 
    if parse_module.isAnyXMLOrAnyDataNode(yangElemNode):
        value["raw_XML_data"] = True
    else:
        value["raw_XML_data"] = False
            
    if parse_module.isLeafNode(yangElemNode):
        value["leaf_node"] = True
    else:
        value["leaf_node"] = False
    
    if parse_module.isListNode(yangElemNode):
        value["list_node"] = True
    else:
        value["list_node"] = False
            
            
    if parse_module.isLeafListNode(yangElemNode):
        value["leaflist_node"] = True
    else:
        value["leaflist_node"] = False
         
    if parse_module.isListKeyNode(yangElemNode):         
        value["listkey_node"] = True
    else:
        value["listkey_node"] = False     
          
          
    if parse_module.isPreseceContainerNode(yangElemNode):
        value["prensence_container_node"] = True
    else:
        value["prensence_container_node"] = False     
            
    operationVal = node.get("operation")       
    if operationVal is not None and operationVal != "unspecified":
        value["operation"] = operationVal
        
    withDefaultVal = node.get("default-attribute")     
    if withDefaultVal is not None and withDefaultVal != "unspecified":  
        value["with_default"] = withDefaultVal == "true"
        
    yangInsertOptionVal = node.get("ordered-by-user")
    if yangInsertOptionVal is not None and yangInsertOptionVal != "unspecified":  
        value["insert_operation"] = yangInsertOptionVal

    yangInsertBeforeAfterTargetVal = node.get("ordered-by-user-insert-list-value")
    if yangInsertBeforeAfterTargetVal is not None:  
        # yangInsertBeforeAfterTargetVal returned is key1=val1;key2=val2...
        keyNameValList=[]
        keyNameValStrList = yangInsertBeforeAfterTargetVal.split(";")
        for keyNameValStr in keyNameValStrList:
            keyName, keyVal = keyNameValStr.split("=")
            keyNameValList.append((keyName, keyVal))
        value["target_key"] = keyNameValList

    yangInsertBeforeAfterTargetVal = node.get("ordered-by-user-insert-leaf-list-value")
    if yangInsertBeforeAfterTargetVal is not None:
        # yangInsertBeforeAfterTargetVal returned is a value  
        value["target_value"] = yangInsertBeforeAfterTargetVal                
    return ((nodeID, value), mymap)

def getInstanceIDPath(node, rootNode):    
    # this will return ID with namespace qualified for 
    # node that is augmented by  another module    
    # /id1/id1/{augmenting-namespace}id3
    # the '/' in augmenting-namespace is replaced with '|' 
    # so splitting can work
    # when referencing imported node
    #  /id1/pfx:id2/id3
    instanceTreeRoot = getInstanceTreeRoot(node, rootNode)
    nodeID = ""
    
    while True and node is not None:
        myID = _get_prefix_qualified_name_for_imported(node, instanceTreeRoot)
        DEBUGPRINT("\tgetInstanceIDPath myID ", myID, " CURRENT nodeID ", nodeID)
        if nodeID != "":                
            nodeID =  "/" + myID +  nodeID
        else:
            nodeID =  "/" + myID 
        node = node.getparent()
        if node is rootNode:
            break
    DEBUGPRINT("\tgetInstanceIDPath nodeID ", nodeID)    
    return nodeID
                            
def _get_prefix_qualified_name_for_imported(node, rootElem):    
    rootNameSpace = etree.QName(rootElem).namespace
    #DEBUGPRINT("rootNameSpace ", rootNameSpace)
    qname = etree.QName(node)
    namespace = qname.namespace
    localname = qname.localname
       
    for pfx, ns in list(node.nsmap.items()):
        #DEBUGPRINT("_get_prefix_qualified_name_for_imported pfx ", pfx, " namespace ", ns,
        #  "compare to ", namespace, " rootNameSpace ", rootNameSpace)
        if namespace == ns:
            if pfx is None:
                if ns == rootNameSpace:
                    return localname
                else:
                    # use qualified name to indicate it's a node defined 
                    # in other module
                    # i.e. node in augmenting module
                    # replace / with |, assuming |  will not appears in a 
                    # namespace string                       
                    #return re.sub(r"/",r"|||", "{%s}%s"%(ns, localname))                    
                    return  "{%s}%s"%(ns, localname)
                    #return  "{%s}%s"%(ns, localname)
                    #return  localname
                
            elif ns == rootNameSpace:
                return localname   
            else:    
                return "%s:%s"%(pfx, localname)

def getInstanceTreeRoot(node, dataRootNode):
    # Given an instanc node 
    # Find the top instance node just under <data>
    # It corresponds to the top node under a model root 
    #
    # In this case We can't use just
    # node.getroottree().getroot()            
    parent = node
    while True:
        if parent is dataRootNode or parent is None:
            break
        else:
            node = parent       
            parent = node.getparent()     
    return node        
    
def getYangTreeRoot(node, dataRootNode):
    # Given a state or config data instance node
    # Get the corresponding YANG tree root
    # node: a node in instance tree
    # dataRootNode: <data>  
    treeObjRoot = None 
    instanceRoot = getInstanceTreeRoot(node, dataRootNode)    
    namespace = etree.QName(instanceRoot).namespace
    DEBUGPRINT("top namespace ", namespace)
    if namespace in parse_module.SS_YANG_Dict:
        treeObjRoot = parse_module.SS_YANG_Dict[namespace]["lxmlTreeRoot"]        
    return treeObjRoot

def getCorrespondingYangFile(node, dataRootNode):    
    instanceRoot = getInstanceTreeRoot(node, dataRootNode)    
    # node is the 'top' level node under <data>
    
    namespace = etree.QName(instanceRoot).namespace
    #localname = etree.QName(node).localname
    
    # the top namespace determines which yang module it is defined in
    # including augmented nodes by other yang modules
    
    if namespace in parse_module.SS_YANG_Dict:                     
        return parse_module.SS_YANG_Dict[namespace]["file"]
    else:
        return "Not Found!"

def _isEditableNode(node, rootNode, session):
    yangElemNode = getInstanceNodeYangElem(node, rootNode, session)         
    if yangElemNode is not None:
        statement = parse_module.getStatementValue(yangElemNode)
        if statement in _dataNodesNoValue:
            return False
        return True
    return False

def isAnyXMLOrAnyDataNode(node, rootNode, session, 
        foundTreeRoot = None, ssYangTreeInst = None):
    yangElemNode = getInstanceNodeYangElem(node, rootNode)         
    return parse_module.isAnyXMLOrAnyDataNode(yangElemNode)     
    
def isLeafList(node, rootNode, session,
        foundTreeRoot = None, 
        ssYangTreeInst = None):
    # rootNode: in the instance tree where this node is not
    #  a node in yang tree such as <reply<, <data> etc.
    yangElemNode = getInstanceNodeYangElem(node, rootNode)         
    if parse_module.getStatementValue(yangElemNode) == "leaf-list":
        return True
    else:
        return False
                                  
def _isList(node, rootNode):
    # rootNode: in the instance tree where this node
    # is not a node in yang tree such as <reply<, <data> etc.
    yangElemNode = getInstanceNodeYangElem(node, rootNode)         
    if parse_module.getStatementValue(yangElemNode) == "list":
        return True
    else:
        return False
                              
def _getListKey( node, rootNode):
    yangElemNode = getInstanceNodeYangElem(node,
             rootNode)
    return parse_module.getListKey(yangElemNode)
                              
def _isListKey( node, rootNode):  
    yangElemNode = getInstanceNodeYangElem(node, 
        rootNode)        
    return parse_module.isListKeyNode(yangElemNode)     

   
