#---------------------------------------------------------------------
# Copyright (C) 2013  Seguesoft  Inc.
#                                                                             
# Redistribution of this software, in whole or in part, is prohibited         
# without the express written permission of Seguesoft. 
# Modified based on ncclient
# ----------------------------------------------------------------------
# Copyright 2009 Shikhar Bhushan
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from ..xml_ import *

from .rpc import RPC

from . import util

import logging

logger = logging.getLogger("ncclient.operations.edit")

"Operations related to changing device configuration"

class EditConfig(RPC):
    "`edit-config` RPC"

    def request(self, target, config, default_operation=None, test_option=None, error_option=None,msgid=None,forceUrl=False):
        """Loads all or part of the specified *config* to the *target* configuration datastore.

        *target* is the name of the configuration datastore being edited

        *config* is the configuration, which must be rooted in the `config` element. It can be specified either as a string or an :class:`~xml.etree.ElementTree.Element`.

        *default_operation* if specified must be one of { `"merge"`, `"replace"`, or `"none"` }

        *test_option* if specified must be one of { `"test_then_set"`, `"set"` }

        *error_option* if specified must be one of { `"stop-on-error"`, `"continue-on-error"`, `"rollback-on-error"` }

        The `"rollback-on-error"` *error_option* depends on the `:rollback-on-error` capability.
        """
        node = new_ele("edit-config")
        node.append(util.datastore_or_url("target", target, self._assert, forceUrl=forceUrl))         
        
        #
        #Input parameters are encoded as child XML elements to the rpc node's
        #XML element, in the same order as they are defined within the "input"
        #statement.
        #This usually does not matter since server treat them as keyword arguments

        if default_operation is not None:
        # TODO: check if it is a valid default-operation
            sub_ele(node, "default-operation").text = default_operation
            
        if test_option is not None:
            #self._assert(':validate')
            sub_ele(node, "test-option").text = test_option

        if error_option is not None:
            #if error_option == "rollback-on-error":
            #    self._assert(":rollback-on-error")
            sub_ele(node, "error-option").text = error_option
            
        spec =validated_element(config, ("config", qualify("config"), "url",qualify("url")))    

        node.append(spec)
        return self._request(node, msgid=msgid)

class EditData(RPC):
    "`edit-data` RPC"

    def request(self, datastore, config, default_operation=None, msgid=None):
        """Loads all or part of the specified *config* to the *target* configuration datastore.

        *target* is the name of the configuration datastore being edited

        *config* is the configuration, which must be rooted in the `config` element. It can be specified either as a string or an :class:`~xml.etree.ElementTree.Element`.

        """
        NS_MAP= {None  : "urn:ietf:params:xml:ns:yang:ietf-netconf-nmda",
                "ds" : "urn:ietf:params:xml:ns:yang:ietf-datastores"}
        node = new_ele("{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}edit-data", 
                            nsmap=NS_MAP)
        # datastore                    
        snode = sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}datastore")
        snode.text = "ds:%s"%datastore

        if default_operation is not None:
        # TODO: check if it is a valid default-operation
            sub_ele(node, "default-operation").text = default_operation
        #Input parameters are encoded as child XML elements to the rpc node's
        #XML element, in the same order as they are defined within the "input"
        #statement.
        #This usually does not matter since server treat them as keyword arguments
           
        spec =validated_element(config, ("config", qualify("config"), "url",qualify("url")))    

        node.append(spec)
        return self._request(node, msgid=msgid)

class DeleteConfig(RPC):
    "`delete-config` RPC"

    def request(self, target, msgid=None,forceUrl=False):
        """Delete a configuration datastore.

        *target* specifies the  name or URL of configuration datastore to delete

        :seealso: :ref:`srctarget_params`"""
        node = new_ele("delete-config")
        node.append(util.datastore_or_url("target", target, self._assert, forceUrl=forceUrl))
        return self._request(node, msgid=msgid)


class CopyConfig(RPC):
    "`copy-config` RPC"

    def request(self, source, target, withDefaults=None, msgid=None,forceUrl=False):
        """Create or replace an entire configuration datastore with the contents of another complete
        configuration datastore.

        *source* is the name of the configuration datastore to use as the source of the copy operation or `config` element containing the configuration subtree to copy

        *target* is the name of the configuration datastore to use as the destination of the copy operation

        :seealso: :ref:`srctarget_params`"""
        node = new_ele("copy-config")
        node.append(util.datastore_or_url("target", target, self._assert,forceUrl=forceUrl))
        node.append(util.datastore_or_url("source", source, self._assert,forceUrl=forceUrl))
        if withDefaults is not None:
            node.append(util.build_withDefaults(withDefaults))         
        return self._request(node, msgid=msgid)

   
    
class Validate(RPC):
    "`validate` RPC. Depends on the `:validate` capability."

    DEPENDS = [':validate']

    def request(self, source, msgid=None, forceUrl=False):
        """Validate the contents of the specified configuration.

        *source* is the name of the configuration datastore being validated or `config` element containing the configuration subtree to be validated

        :seealso: :ref:`srctarget_params`"""
        #node = new_ele("validate")
        try:
            src = validated_element(source, ("config", qualify("config")))
        except Exception as e:
            logger.debug(e)
            src = util.datastore_or_url("source", source, self._assert,forceUrl=forceUrl)
        if not source.startswith("nmda-"):    
            node = new_ele("validate")
            (node if src.tag == "source" else sub_ele(node, "source")).append(src)
        else:
            _, datastore = source.split("-")
            DS_NSMAP = {"ds" : "urn:ietf:params:xml:ns:yang:ietf-datastores",            
                        "nmda" : "urn:ietf:params:xml:ns:yang:ietf-netconf-nmda"}
            node = new_ele("validate", nsmap= {"ds" : "urn:ietf:params:xml:ns:yang:ietf-datastores",            
                        "nmda" : "urn:ietf:params:xml:ns:yang:ietf-netconf-nmda"})
            tnode = sub_ele(node, "source")
            
            snode = sub_ele(tnode, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}datastore")
            snode.text = "ds:%s"%datastore

        return self._request(node, msgid=msgid)


class Commit(RPC):
    "`commit` RPC. Depends on the `:candidate` capability, and the `:confirmed-commit`."

    DEPENDS = [':candidate']

    def request(self, confirmed=False, timeout=None, persist=None, persistid=None, msgid=None):
        """Commit the candidate configuration as the device's new current configuration. Depends on the `:candidate` capability.

        A confirmed commit (i.e. if *confirmed* is `True`) is reverted if there is no followup commit within the *timeout* interval. If no timeout is specified the confirm timeout defaults to 600 seconds (10 minutes). A confirming commit may have the *confirmed* parameter but this is not required. Depends on the `:confirmed-commit` capability.

        *confirmed* whether this is a confirmed commit

        *timeout* specifies the confirm timeout in seconds"""

        node = new_ele("commit")
        if confirmed:
            #self._assert(":confirmed-commit")
            sub_ele(node, "confirmed")
            if timeout is not None:
                sub_ele(node, "confirm-timeout").text = timeout
            if persist is not None:
                #must check if the server advertise 1.1
                #self._assert(":confirmed-commit:1.1")
                sub_ele(node, "persist").text = persist
                
        if persistid is not None:
            #must check if the server advertise 1.1
            #self._assert(":confirmed-commit:1.1")
            sub_ele(node, "persist-id").text = persistid
                
        return self._request(node, msgid=msgid)


class DiscardChanges(RPC):
    "`discard-changes` RPC. Depends on the `:candidate` capability."

    DEPENDS = [":candidate"]

    def request(self, msgid=None):
        """Revert the candidate configuration to the currently running configuration. Any uncommitted changes are discarded."""
        return self._request(new_ele("discard-changes"), msgid=msgid)
        
class CancelCommit(RPC):
    "`cancel-commit` RPC. Depends on the `:candidate 1.1` capability."

    DEPENDS = [":candidate"]

    def request(self, persistid=None, msgid=None):
        """Cancels an ongoing confirmed commit. If persistid is given cancels a persistent confirmed commit."""
        node = new_ele("cancel-commit")
        if persistid is not None:
            #self._assert(":confirmed-commit")
            sub_ele(node, "persist-id").text = persistid
        return self._request(node, msgid=msgid)
            
        
class GetSchema(RPC):
    "`get-schema` RPC. Depends on the `:candidate` capability."

    def request(self, identifier=None, version=None, format=None, msgid=None):
        """Download schema"""
        node = new_ele("get-schema", xmlns=NC_MONITORING)
        if identifier is not None:
            sub_ele(node, "identifier").text = identifier
        if version is not None:
            sub_ele(node, "version").text = version
        if format is not None:
            sub_ele(node, "format").text = format
            
        return self._request(node, msgid=msgid)
        
        
