#---------------------------------------------------------------------
# Copyright (C) 2013  Seguesoft  Inc.
#                                                                             
# Redistribution of this software, in whole or in part, is prohibited         
# without the express written permission of Seguesoft. 
# Modified based on ncclient
# ----------------------------------------------------------------------
# Copyright 2009 Shikhar Bhushan
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from pprint import PrettyPrinter
from .rpc import RPC, RPCReply

from ..xml_ import *

from . import util

class GetReply(RPCReply):

    """Adds attributes for the *data* element to `RPCReply`."""

    def _parsing_hook(self, root):
        self._data = None
        if not self._errors:
            self._data = root.find(qualify("data"))
            if self._data is None:
                self._data = root.find(qualify("data", ns="urn:ietf:params:xml:ns:yang:ietf-netconf-nmda"))

    @property
    def data_ele(self):
        "*data* element as an :class:`~xml.etree.ElementTree.Element`"
        if not self._parsed:
            self.parse()
        return self._data

    @property
    def data_xml(self):
        "*data* element as an XML string"
        if not self._parsed:
            self.parse()
        return to_xml(self._data, pretty_print=util.indent_flag)
    
    """
    @property
    def request_xml(self):
        "*request* element as an XML string"
        return self.request_xml

    @property
    def request_ele(self):
        "*request* element as an XML string"
        return self.request_ele
    """
    data = data_ele
    "Same as :attr:`data_ele`"


class Get(RPC):

    "The *get* RPC."

    REPLY_CLS = GetReply
    "See :class:`GetReply`."

    def request(self, filter=None, withDefaults=None,xPathNamespacePrefixMap=None, msgid=None):
        """Retrieve running configuration and device state information.

        *filter* specifies the portion of the configuration to retrieve (by default entire configuration is retrieved)

        :seealso: :ref:`filter_params`
        """
        #node = new_ele("get", {}, **{})
        node = new_ele("get")

        
        if filter is not None:
            node.append(util.build_filter(filter, xPathNamespacePrefixMap=xPathNamespacePrefixMap))  
        if withDefaults is not None and withDefaults != "unspecified":
            node.append(util.build_withDefaults(withDefaults)) 
                        
        return self._request(node, msgid=msgid)

class GetData(RPC):

    "The *get-data* RPC."

    REPLY_CLS = GetReply
    "See :class:`GetReply`."

    def request(self, filter=None,   msgid=None,
                    xpath=False, xPathNamespacePrefixMap=None,
                    datastore="running",
                    config= 2,
                    originFilter=[],
                    negatedOriginFilter=[], 
                    maxDepth=0,
                    withOrigin=False, 
                    withDefaults=None
                ):
        """Retrieve running configuration and device state information.

        *filter* specifies the portion of the configuration to retrieve (by default entire configuration is retrieved)
        # config: 2 = unspecified, 1 = config True,  0 = config False
        # originFilter and negatedOriginFilter can be a list of ['intended', 'system', 'dynamic', 'learned', 'default', 'unknown']
        # maxDepth: 0 = unspecified. otherwise "unbounded"  or 1..65535

        :see also: :ref:`filter_params`
        """
        NS_MAP= {None  : "urn:ietf:params:xml:ns:yang:ietf-netconf-nmda",
                "ds" : "urn:ietf:params:xml:ns:yang:ietf-datastores"}

        if len(originFilter)>0 and len(negatedOriginFilter)>0:
            raise Exception("negated-origin-filter and origin-filter can't be used at the same time!")
        if len(originFilter)>0 or len(negatedOriginFilter)>0:
            NS_MAP.update({"or" : "urn:ietf:params:xml:ns:yang:ietf-origin"})

        node = new_ele("{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}get-data", 
                            nsmap=NS_MAP)
        snode = sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}datastore")
        snode.text = "ds:%s"%datastore
        
        if filter is not None:
            node.append(util.build_get_data_filter(filter, xpath=xpath, xPathNamespacePrefixMap=xPathNamespacePrefixMap))  

        if config == 1 or config == True:
            snode= sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}config-filter")
            snode.text= "true"
        elif config == 0 or config == False:
            snode = sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}config-filter")
            snode.text= "false"

        if len(originFilter):
            for orfilter in originFilter:
                snode = sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}origin-filter")
                snode.text = "or:%s"%orfilter
        elif len(negatedOriginFilter):    
            for norfilter in negatedOriginFilter:
                snode = sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}origin-filter")
                snode.text = "or:%s"%norfilter

        if maxDepth != "unspecified" and maxDepth !=0:
            snode = sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}max-depth")
            snode.text =  str(maxDepth)

        if withOrigin:
            sub_ele(node, "{urn:ietf:params:xml:ns:yang:ietf-netconf-nmda}with-origin")

        if withDefaults is not None and withDefaults != "unspecified":
            node.append(util.build_withDefaults(withDefaults)) 
                        
        return self._request(node, msgid=msgid)

class GetConfig(RPC):

    "The *get-config* RPC."

    REPLY_CLS = GetReply
    "See :class:`GetReply`."

    def request(self, source, filter=None, withDefaults=None,
                    xPathNamespacePrefixMap=None, msgid=None):
        """Retrieve all or part of a specified configuration.

        *source* name of the configuration datastore being queried

        *filter* specifies the portion of the configuration to retrieve (by default entire configuration is retrieved)

        :seealso: :ref:`filter_params`"""
        node = new_ele("get-config")
        node.append(util.datastore_or_url("source", source, self._assert))
        if filter is not None:
            node.append(util.build_filter(filter, xPathNamespacePrefixMap=xPathNamespacePrefixMap))
        if withDefaults is not None and withDefaults != "unspecified":
            node.append(util.build_withDefaults(withDefaults))             
        return self._request(node, msgid=msgid)