# -*- coding: utf-8 -*-
# Modified for YANG JSON XML mapping
# Copyright (c) 2017 SegueSoft Inc. 
#
#	This file contains the proprietary intellectual property of
#	SegueSoft, Inc.  
#   
#   Redistribution and use in source or binary
#	forms is not permitted without the express written permission.
#
#   Permission to use this file for the purpose of interfacing 
#	SegueSoft NETCONF+ RESTCONF application to other applications
#   is granted to licensed customers of SegueSoft 
#
#
# Copyright (c) 2015, S Anand
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
# 

import os, sys, ast, traceback
from collections import Counter, OrderedDict

from lxml.etree import Element
from lxml import etree as xmletree
from lxml.etree import cleanup_namespaces

import parse_module
from parse_module import getStatementValue, getArgumentValue
from ssinstance import isLeafList
from ssinstance import _isList
from ssinstance import getInstanceNodeYangElem

from common_global_variables import DEBUGPRINT

from ssyangtree import SSYangTree
import json

sys.path.append(os.path.join(os.path.dirname(os.path.abspath(__file__)) , ".."))

__author__ = 'S Anand'
__email__ = 'root.node@gmail.com'
__version__ = '0.1.9'

# Python 3: define unicode() as str()
#if sys.version_info[0] == 3:
    #str = str
    #str = str


class XMLData(object):
    def __init__(self, session, xml_fromstring=True, xml_tostring=True, element=None, dict_type=None,
                 list_type=None, attr_prefix=None, text_content=None, simple_text=False):
                 
        self.session = session # netconf or restconf session         
        # xml_fromstring == False(y) => '1' -> '1'
        # xml_fromstring == True     => '1' -> 1
        # xml_fromstring == fn       => '1' -> fn(1)
        if callable(xml_fromstring):
            self._fromstring = xml_fromstring
        elif not xml_fromstring:
            self._fromstring = lambda v: v
        # custom conversion function to convert data string to XML string
        if callable(xml_tostring):
            self._tostring = xml_tostring
        # custom etree.Element to use
        self.element = Element if element is None else element
        # dict constructor (e.g. OrderedDict, defaultdict)
        self.dict = OrderedDict if dict_type is None else dict_type
        # list constructor (e.g. UserList)
        self.list = list if list_type is None else list_type
        # Prefix attributes with a string (e.g. '$')
        self.attr_prefix = attr_prefix
        # Key that stores text content (e.g. '$t')
        self.text_content = text_content
        # simple_text == False or None or 0 => '<x>a</x>' = {'x': {'a': {}}}
        # simple_text == True               => '<x>a</x>' = {'x': 'a'}
        self.simple_text = simple_text

    def _tostring(self, value):
        '''Convert value to XML compatible string'''
        if value is True:
            return 'true'
        elif value is False:
            return 'false'
            
        try:
            return str(value)     
        except Exception:
            # All of its arguments should be 8-bit strings.
            # The first argument is converted to Unicode using
            # the specified encoding; if you leave off the encoding argument, 
            # the ASCII encoding is used for the conversion, so characters 
            # greater than 127 will be treated as errors:
            
            return str(value, encoding="utf8")                 
        
    def _convert_prefix_Value_to_modulename(self, value, nsmap):
        modulepfx = ""
        valspt = value.split(":")
        if len(valspt) > 1:
            # value is preifx:label
            # need to find and replace prefix with modulename
            modulepfx, label = valspt                                    
        
        if modulepfx != "":
            if modulepfx in self.session.session_prefix_namespace_dict:
                modulens = self.session.session_prefix_namespace_dict[modulepfx]   
                if modulens in nsmap:
                    modulename = nsmap[modulens]
                    return "%s:%s"%(modulename, label)

        else:    
            return value
            
    
    def _fromstring(self, value, type=None, yangElemNode=None, nsmap={}):
        '''Convert XML string value to None, boolean, int or float (json typed values)
            None == > null
            True/False => true/false
            number  == >unquoted number
            string => double quoted string 
            
        '''
        if not value:
            return None
            
        if yangElemNode is not None and len(yangElemNode) == 0:
            # this must be a node or faked to be a node in  insntance tree
            DEBUGPRINT("before instance yangElemNode ", yangElemNode)
            yangElemNode=getInstanceNodeYangElem(yangElemNode, yangElemNode.getroottree().getroot())
            DEBUGPRINT("corresponding schema yangElemNode ", yangElemNode)
        #std_value = value.strip().lower()
        #if std_value == 'true':
        #    return True
        #elif std_value == 'false':
        #    return False

        if yangElemNode.tag == '{urn:ietf:params:xml:ns:yang:yin:1}input': 
            return ""

        if yangElemNode is not None:
            try:
                foundTreeRoot = yangElemNode.getroottree().getroot() 
                type = parse_module.lookup_property(yangElemNode, "base-type", rootNode=foundTreeRoot)
                DEBUGPRINT("#### ", yangElemNode.tag, " found node type ", type, " ", getArgumentValue(yangElemNode))
            except Exception as e:
                DEBUGPRINT("!!! ", yangElemNode.tag, " " ,e, " traceback: ", traceback.format_exc())
                raise 
        DEBUGPRINT("got value ", value)
        if type is not None:    
            if type == "empty":
                # this will end up to be [null] 
                return [None]
            elif type in ["int8", "int16", "int32", "uint8", "uint16", "uint32"]:
                try:
                    res = int(value)  
                except Exception as e:
                    # use whatever is passed in (mainly for testing like send str for int)
                    res = value
                return res
            elif type == "boolean":
                std_value = value.strip().lower()
                if std_value == 'true':
                    return True
                elif std_value == 'false':
                    return False
                else:    
                    return False
            #elif type == "union":
            #    #value=""
            #    return value
            else:
                # decimal64 , int64, string, bits, enumeration, "leaf-ref", identityref etc.
                #value ="想想"
                try:
                    resval= str(value)
                except Exception as e:
                    resval = str(value, encoding="utf8")
                                        
                if type == "identityref":    
                    resval = self._convert_prefix_Value_to_modulename(resval, nsmap)    
                                        
                return resval
        # no matching element found just return the untyped value as string
        return value
        
    def _convert_attr(self, attrib, nsmap):
        DEBUGPRINT("passed in ", attrib, " type", type(attrib))
        attr = {}
        for k, v in list(attrib.items()):
            #print " k ", k , " v ", v            
            ktag = xmletree.QName(k)         
            #print " ktag ", ktag.namespace          
            try:
                kprefix=  nsmap[ktag.namespace]
            except:    
                raise Exception("Unknown namespace %s used in attribute in nsmap!"%ktag.namespace )

            #print " krefix ", kprefix      
            module_qualified_name="%s:%s"%(kprefix, ktag.localname)
            try:
                if module_qualified_name == "ietf-netconf-with-defaults:default":
                    t="boolean"
                else:
                    t = self.session.yang_lookup(module_qualified_name, "base-type")
            except Exception as e:
                # if there is no type defined or not found 
                t = "string"
            #print "for ", module_qualified_name, " type found ", t  
            attr[module_qualified_name] = self._fromstring(v,  type=t, nsmap=nsmap)
            #attr["%s:%s"%(kprefix, ktag.localname)] = v
        return attr
        
    def _set_elem_attr(self, result, attkey, attval, nsmap):
        #print "attkey ", attkey, "attval :", attval, " json ", json.dumps(attval)
        modulename = ""
        keyspt = attkey.split(":")
        if len(keyspt) > 1:
            # key is modulename:label
            # need to insert namespace
            modulename, label = keyspt                                    
        
        if modulename != "":
            if modulename in nsmap:
                moduelns = nsmap[modulename]    
                #root is result here
                result.set("{%s}%s"%(moduelns, label), self._tostring(attval))
            else:
                raise Exception("Unknown nsmap for %s!"%modulename)
        else:    
            usens = xmletree.QName(result).namespace         
            result.set("{%s}%s"%(usens, attkey), self._tostring(attval))    
            
            
    def etree(self, data, root=None, nsmap={}):
        '''Convert data structure into a list of etree.Element'''
        result = self.list() if root is None else root
        if isinstance(data, (self.dict, dict)):
            key = ""
            for key, value in list(data.items()):
                #print "key: ", key, " VALUE ", value
                value_is_list = isinstance(value, (self.list, list))
                value_is_dict = isinstance(value, (self.dict, dict))               
                
                # Add attributes and text to result (if root)
                if root is not None:
                    # Handle attribute prefixes (BadgerFish)
                    if self.attr_prefix is not None:
                        if key.startswith(self.attr_prefix):
                            key = key.lstrip(self.attr_prefix)                            

                            # Annotations to anydata, container, and list Entries
                            # For a data node instance that is encoded as a JSON object (i.e., a
                            # container, list entry, or anydata node), the metadata object is added
                            # as a new member of that object with the name "@".
                            #print "Annotations to anydata, container, and list Entries: ", value  
                            # e.g.:
                            # "@": {
                            # "example-last-modified:last-modified": "2015-09-16T10:27:35+02:00"
                            #}    
                            
                            try:     
                                #self._set_elem_attr(result, k, v, nsmap)
                                if key == "":
                                    #print "Annotations to anydata, container, and list Entries: ", value  
                                    # e.g.:
                                    # "@": {
                                    # "example-last-modified:last-modified": "2015-09-16T10:27:35+02:00"
                                    #}    
                                    for k, v in list(value.items()):
                                        self._set_elem_attr(result, k, v, nsmap)
                                        
                                else:
                                    # temporary added to the parent node , later will be moved to the appropriate 
                                    # child nodes (json object maintains no order so the attrtibute (metadata) 
                                    # node may get processded earlier than the actual data node
                                    if value_is_list:
                                        # Annotations to leaf-list Instances
                                        #print "Annotations to leaf-list Instances ", " key ", key, " value ", value
                                        # temporary added to the parent node , later will be moved to the appropriate 
                                        # child nodes (json object maintains no order so the attrtibute (metadata) 
                                        # node may get processded earlier than the actual data node
                                        self._set_elem_attr(result, key, value, nsmap)
                                        
                                    else:                                    
                                        # Annotations to anyxml and leaf Instances,
                                        # or generic XML attribute 
                                        # such as RFC8040 section 5.3.1 wd:default="true"                                    
                                        #print "Annotations to anyxml and leaf Instances, or generic XML attribute",  " key ", key, " value ", value
                                        self._set_elem_attr(result, key, value, nsmap)
 
                            except Exception as e:
                                raise 
                            
                            continue
                    # Handle text content (BadgerFish, GData)
                    if self.text_content is not None:                        
                        if key == self.text_content:
                            # key '$' was added previosuly as key to value                            
                            result.text = self._tostring(value)
                            continue
                            
                    # BadgerFish attr_prefix is not None so the following won't be applicable
                    # 
                    # Treat scalars as text content, not children (GData)
                    #if self.attr_prefix is None and self.text_content is not None:
                    #    if not value_is_dict and not value_is_list:
                    #        result.set(key, self._tostring(value))
                    #        continue
                
                # Add keys in *value* as one or more children
                values = value if value_is_list else [value]
                usens = xmletree.QName(result).namespace  
                for value in values:                   
                    modulename = ""
                    keyspt = key.split(":")
                    if len(keyspt) > 1:
                        # key is modulename:label
                        # need to insert namespace
                        modulename, key = keyspt
                        
                        
                    # The following code will generate a tree with all node name
                    # qualified by a prefix
                    if modulename != "":
                        if modulename in nsmap:
                            # do not use the following hack, it does assign namespce
                            # to the node until the constructed xml is read by fromstring()
                            #elem.attrib["xmlns"] = nsmap[modulename]
                            usens = nsmap[modulename]
                            DEBUGPRINT("usens : ", usens)
                            elem = self.element("{%s}%s"%(usens, key), nsmap=self.session.moduleToNamespaceImplementedDict)
                            cleanup_namespaces(elem)
                        else:
                            raise Exception("Unknown nsmap %s!"%modulename)
                    else:
                        #usens = xmletree.QName(result).namespace  
                        # recursive ... so we use the most recent usens at each recursion.
                        # e.g. get on 'module' node of 
                        #DEBUGPRINT("usens 2 : ", usens)       
                        elem = self.element("{%s}%s"%(usens, key), nsmap=nsmap)
                        cleanup_namespaces(elem)
                                            
                    result.append(elem)
                    
                    # Treat scalars as text content (make it an object  {"$" :value}, not children
                    # when calling this function next time '$' key will be processed specially
                    #
                    # json value can be of type: (array, object (dict), or scalar)
                    if not isinstance(value, (self.dict, dict, self.list, list)):
                        if self.text_content:
                            value = {self.text_content: value}
                    self.etree(value, root=elem, nsmap=nsmap)
                
            # after processing all immediate children of data   
            # move attributes set earlier in the root (parent) to the relevant child.
            # do not perform attribute check for '$' text content key added while 
            # scanning data dict

            if key != "" and key != self.text_content:
                for r in result.iterchildren():
                    if r.tag in result.attrib:
                        #print "found r.tag ", r.tag, " => ", result.attrib[r.tag], " type", type(result.attrib[r.tag])
                        x = ast.literal_eval(result.attrib[r.tag])

                        if isinstance(x, (self.list, list)):
                            # leaf-list attr
                            # e.g. [None, {u'example-last-modified:last-modified': u'2015-06-18T17:01:14+02:00'}] 
                            for xe in x:
                                if xe is None:
                                    # break inner loop and continue next outer loop
                                    # looking for next leaf-list instance 
                                    x.remove(xe)
                                    result.set(r.tag, self._tostring(x))
                                    #result.set(r.tag, json.dumps(x))
                                    break
                                else:
                                    # {u'ietf-netconf-with-defaults:default': True} is a dict
                                    #print "????????????x again result.attrib[r.tag] type ", xe, " type ",type(xe)    
                                    for k, v in list(xe.items()):
                                        self._set_elem_attr(r, k, v, nsmap)
                                    # make sure we don't repeat processing this one        
                                    x.remove(xe)
                                    result.set(r.tag, self._tostring(x))
                                    #result.set(r.tag, json.dumps(x))
                                        
                        else:
                            # anyxml and leaf Instances
                            # e.g.: a:mtu="{u'ietf-netconf-with-defaults:default': True}"
                            # {u'ietf-netconf-with-defaults:default': True} is a dict
                            # 
                            #print "result.attrib[r.tag] type ", x, " type ",type(x)    
                            for k, v in list(x.items()):
                                self._set_elem_attr(r, k, v, nsmap)
                            # remove from parent node                 
                            #del result.attrib[r.tag]       
                # finally remove from parent node                 
                for r in result.iterchildren():
                    if r.tag in result.attrib:
                        del result.attrib[r.tag]    
                            
        else:
            if self.text_content is None and root is not None:
                root.text = self._tostring(data)
            else:
                result.append(self.element(self._tostring(data), nsmap=nsmap))
        return result

        
                       
    def data(self, root, nsmap={}):
        '''Convert etree.Element into a dictionary'''    
        value = self.dict()
        children = [node for node in root if isinstance(node.tag, str)]
        
        roottag = xmletree.QName(root)         
        DEBUGPRINT("process new root ", roottag, " ns ", roottag.namespace)
        try:
            rootprefix=  nsmap[roottag.namespace]
        except:    
            raise Exception("Unknown namespace %s in nsmap!"%roottag.namespace )
                
        for attr, attrval in list(root.attrib.items()):
            if len(root):
                value.setdefault("@", {});
                # this attribute must be for a container , list entry , anydata
                # must be a container , list entry , anydata
                attrtag = xmletree.QName(attr)         
                try:
                    attrprefix=  nsmap[attrtag.namespace]
                except:    
                    raise Exception("Unknown attribute namespace %s in nsmap!"%attrtag.namespace )
                
                value["@"].update(self.dict([("%s:%s"%(attrprefix,attrtag.localname ), attrval)]))
            else:
                # leaf anyxml etc.
                # leaf-list
                # this should be handled when process the parent of 'root' node
     
                pass
                # create a sibling element
               
                
        # scan the grand child of root and check if they have attributes
        # if they have we need to insert new child nodes
        insert_nodes ={}
        for child in children:
            if len(child) > 0 :
                # here we may need to check isContainer , is anyData?
                # this attribue must be for a container , list entry , anydata
                # they are handled when process its child recursilvely 
                continue 
            
            insert_nodes.setdefault(child.tag, [])
            if len(child.attrib):
                # have attributes           
                # if there are more than one child with the same tag the it is a leaf-list 
                DEBUGPRINT("for child.tag ", child.tag, " leaf or leaf-list metadata: ", attr)
                # leaf or single item leaf-lsit (we may have to call is-leaf-list in future)
                insert_nodes[child.tag].append(self._convert_attr(child.attrib, nsmap))
            else:    
                # leaf-list need this)
                # leaf or single item leaf-list (we may have to call is-leaf-list in future)
                insert_nodes[child.tag].append([])      
                    
        # remove child nodes that do not have any attribtues
        pruned_insert_nodes = {}
        for k, v in list(insert_nodes.items()):
            DEBUGPRINT("check for root ", root.tag, " k ", k , " v ", v, " ==None?: ", v == [[]])
            if v!= [[]]:
                av=[]
                for a in v:
                    if a == []:
                        #av=None
                        av.append(None)
                    else:
                        av.append(a)
                pruned_insert_nodes[k] = av

        insert_nodes = pruned_insert_nodes
        DEBUGPRINT("insert_nodes ", insert_nodes)
        # handle the value of the root in this level of the recursive data() call   
        if root.text and self.text_content is not None:
            text = root.text.strip()
            if text:
                if self.simple_text and len(children) == 0:
                    value = self._fromstring(text, yangElemNode=root, nsmap=nsmap)
                else:
                    value[self.text_content] = self._fromstring(text, yangElemNode=root, nsmap=nsmap)
                    
        count = Counter(child.tag for child in children)
        for child  in children:
            childtag = xmletree.QName(child)         
            DEBUGPRINT("     process child ", child.tag, "count ", count[child.tag], " value ", getArgumentValue(child) )   
            try:
                childprefix=  nsmap[childtag.namespace]
            except:    
                raise Exception("Unknown child namespace %s in nsmap!"%childtag.namespace )

            DEBUGPRINT("rootprefix ", rootprefix)
            DEBUGPRINT("childprefix ", childprefix)
                        
            if rootprefix == childprefix:
                useChildPrefix = ""
            else:
                useChildPrefix = childprefix + ":"
            DEBUGPRINT("useChildPrefix ", useChildPrefix)
                
            if count[child.tag] == 1 and not _isList(child, child.getroottree().getroot()):                
                value.update(self.data(child, nsmap=nsmap))
            else:
                # Child is also a list
                    
                # (python passes lists by reference) so although result 
                # is not used /referenced the content has been changed                 
                # result is a list of ordered dictionary
                
                # setdefault: If key is in the dictionary, return its value. 
                # If not, insert key with a value of default and return default. 
                # default defaults to None.
                
                result = value.setdefault(useChildPrefix + childtag.localname, self.list())
                # + list concatenation 
                result += list(self.data(child, nsmap=nsmap).values())   
            
            # check if we need to insert meta data entries
            if child.tag in insert_nodes:
                DEBUGPRINT("\n\n\n###for root ", root.tag, " need to add ", child.tag, " =>",
                     insert_nodes[child.tag], " type ", type(insert_nodes[child.tag]), 
                     "len ", len(insert_nodes[child.tag]))
                if len(insert_nodes[child.tag]) == 1:
                    value["@"+useChildPrefix + childtag.localname] = insert_nodes[child.tag][0]
                else:
                    value["@"+useChildPrefix + childtag.localname] = insert_nodes[child.tag]                    
           
                del insert_nodes[child.tag]
                                    
        # decide if we need to qualify with  modulename for the returned dict
        parent = root.getparent()
        if parent  is not None:
            parenttag = xmletree.QName(parent)         
            try:
                parentprefix=  nsmap[parenttag.namespace]
            except:    
                raise Exception("Unknown parent node namespace %s in nsmap!"%parenttag.namespace )        
        else:
            parentprefix = None
            
        #print " parentprefix " ,parentprefix    
        # attribute '@' does not need to be prefixed with module name
        if rootprefix == "" or rootprefix is None or parentprefix == rootprefix or roottag.localname == "@":
            useRootPrefix = ""
        else:
            useRootPrefix = rootprefix + ":"
        return self.dict([(useRootPrefix + roottag.localname, value)])


class RestConfYANG(XMLData):
    '''Converts between XML and data using the Restconf convention'''
    def __init__(self, session, **kwargs):
        super(RestConfYANG, self).__init__(session, attr_prefix='@', text_content='$', simple_text=True, **kwargs)

        

